package cn.com.duiba.biz.tool.duiba.client;

import cn.com.duiba.biz.tool.duiba.dto.ConsumerCookieDto;
import cn.com.duiba.consumer.center.api.dto.ConsumerDto;
import cn.com.duiba.consumer.center.api.remoteservice.RemoteConsumerService;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.idmaker.service.api.remoteservice.kms.RemoteKmsService;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import com.alibaba.fastjson.JSONObject;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;

/**
 * 使用方法，在spring配置文件中声明即可:<bean id="requestLocal" class="cn.com.duiba.biz.tool.duiba.client.RequestLocal"/>
 * 实例化时会自动寻找RemoteAppService、RemoteConsumerService并注入，找不到则报错要求声明这两个bean
 *
 * Created by wenqi.huang on 2017/5/12.
 */
public class RequestLocal implements ApplicationContextAware, InitializingBean{

    private static final Logger logger = LoggerFactory.getLogger(RequestLocal.class);

    private ApplicationContext applicationContext;

    private static RemoteConsumerService remoteConsumerService;
    private static RemoteAppService remoteAppService;
    private static DuibaConsumerCookieClient duibaConsumerCookieClient;

    //这个是用于兼容老的固定密钥加解密逻辑
    @Deprecated
    private static String oldConsumerEncryptKey;

    private static ThreadLocal<HttpRequestDto> local = new ThreadLocal<>();

    private static class HttpRequestDto {

        private ConsumerDto consumerDO;
        private AppSimpleDto consumerAppDO;
        private ConsumerCookieDto consumerCookieDto;//cookie中直接获取的信息
        private String tokenId;
        private HttpServletRequest request;
        private HttpServletResponse response;

        public void init(HttpServletRequest request, HttpServletResponse response){
            this.request = request;
            this.response = response;
            try {
                this.consumerCookieDto = duibaConsumerCookieClient.getConsumerCookieDto(request);
            }catch(Exception e){
                logger.error("get user from wdata4 error,will try to use wdata3.", e);
            }
            if(this.consumerCookieDto == null){//如果取不到用户信息，则走老的逻辑
                parseOldWdata3();
            }

            this.tokenId = RequestTool.getCookie(request, DuibaConsumerCookieClient.TOKEN_ID_COOKIE);
            //TODO to be deleted
            if(this.tokenId == null || this.tokenId.isEmpty()){
                this.tokenId = RequestTool.getCookie(request, "wdata3");
            }
        }

        //解析旧的用户cookie数据
        @Deprecated
        private void parseOldWdata3(){
            String wdata3 = RequestTool.getCookie(request, "wdata3");
            if(wdata3 == null || wdata3.isEmpty()){
                return;
            }
            String content = BlowfishUtils.decryptBlowfish(wdata3, oldConsumerEncryptKey);
            ConsumerCookieDto c = JSONObject.parseObject(content, ConsumerCookieDto.class);
            long now = System.currentTimeMillis();
            if(c.getTime() < now - 86400000){
                return;
            }
            this.consumerCookieDto = c;
        }

        //延迟初始化
        public ConsumerDto getConsumerDO(){
            if(consumerDO == null){
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
                    ConsumerDto temp = new ConsumerDto();
                    temp.setAppId(consumerCookieDto.getAppId());
                    temp.setId(consumerCookieDto.getCid());
                    temp.setPartnerUserId(consumerCookieDto.getPartnerUserId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new ConsumerMethodInterceptor(this));
                    consumerDO = (ConsumerDto)proxyFactory.getProxy();
                }
            }

            return consumerDO;
        }

        //延迟初始化
        public AppSimpleDto getConsumerAppDO() {
            if (consumerAppDO == null) {
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取id字段时不会读取数据库，获取其他字段才会从数据库加载
                    AppSimpleDto temp = new AppSimpleDto();
                    temp.setId(consumerCookieDto.getAppId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new AppMethodInterceptor(this));
                    consumerAppDO = (AppSimpleDto)proxyFactory.getProxy();
                }
            }
            return consumerAppDO;
        }

        public HttpServletResponse getResponse() {
            if(response == null){
                throw new IllegalStateException("response must not be null, please invoke RequestLocal.setThreadLocally first");
            }
            return response;
        }
        public HttpServletRequest getRequest() {
            if(request == null){
                throw new IllegalStateException("request must not be null, please invoke RequestLocal.setThreadLocally first");
            }
            return request;
        }

        public String getTokenId() {
            return tokenId;
        }
    }

    private static class ConsumerMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public ConsumerMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId") || m.getName().equals("getAppId") || m.getName().equals("getPartnerUserId")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            ConsumerDto originalConsumer = (ConsumerDto)invocation.getThis();
            ConsumerDto consumerInDb = remoteConsumerService.find(originalConsumer.getId());

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(consumerInDb, originalConsumer);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerDO = originalConsumer;

            return invocation.proceed();
        }
    }

    private static class AppMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public AppMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            AppSimpleDto originalApp = (AppSimpleDto)invocation.getThis();
            AppSimpleDto appInDb = remoteAppService.getSimpleApp(originalApp.getId()).getResult();

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(appInDb, originalApp);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerAppDO = originalApp;

            return invocation.proceed();
        }
    }

    private static HttpRequestDto get() {
        HttpRequestDto rl = local.get();
        if (rl == null) {
            rl = new HttpRequestDto();
            local.set(rl);
        }
        return rl;
    }

    /**
     * 获得当前登录的用户id
     * @return
     */
    public static Long getCid(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getCid();
    }

    /**
     * 获得当前登录的用户的appId
     * @return
     */
    public static Long getAppId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getAppId();
    }

    /**
     * 获得当前登录的用户的partnerUserId
     * @return
     */
    public static String getPartnerUserId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getPartnerUserId();
    }

    /**
     * 获取当前登录的用户信息,获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static ConsumerDto getConsumerDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerDO();
    }

    /**
     * 获取当前登录的用户所在的app信息,获取id字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static AppSimpleDto getConsumerAppDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerAppDO();
    }

    /**
     * 获取tokenId， tokenId的生命周期和wdata4相同，这个tokenId是唯一可以在js里直接获得的用户相关的cookie，（其他用户cookie全都设为了httpOnly，防止被XSS攻击获得）；作用主要有：用于传递给同盾做会话id。
     * @return
     */
    public static String getTokenId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.getTokenId();
    }

    /**
     * 当前用户是否为登录状态(如果用户的partnerUserId为not_login也视为未登录状态)
     * @return
     */
    public static boolean isLoginConsumer(){
        String partnerUserId = getPartnerUserId();
        return partnerUserId != null && !ConsumerDto.NOTLOGINUSERID.equals(partnerUserId);
    }

    /**
     * 当前用户是否为分享用户
     * @return
     */
    public static boolean isShareConsumer(){
        return ConsumerDto.SHAREUSERID.equals(getPartnerUserId());
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @param cookieDomain cookie需要设置的domain
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer, String cookieDomain){
        if(cookieDomain == null || cookieDomain.isEmpty()){
            throw new IllegalArgumentException("cookieDomain must not be empty");
        }
        if(cookieDomain.startsWith(".") || cookieDomain.startsWith("-")){//不能以.或者-开头
            cookieDomain = cookieDomain.substring(1);
        }

        HttpRequestDto requestLocal = get();

        ConsumerCookieDto consumerCookieDto = duibaConsumerCookieClient.injectConsumerInfoIntoCookie(consumer, requestLocal.getRequest(), requestLocal.getResponse(), cookieDomain, oldConsumerEncryptKey);

        //重置当前登录用户的属性
        requestLocal.consumerCookieDto = consumerCookieDto;
        requestLocal.consumerDO = null;
        requestLocal.consumerAppDO = null;
        requestLocal.tokenId = (String)requestLocal.getRequest().getAttribute("c_tokenId");
    }

    /**
     * 设置request和response ，必须在获取用户信息之前调用（一般是在LoginFilter开始时）
     * @param request
     * @param response
     */
    public static void setThreadLocally(HttpServletRequest request, HttpServletResponse response){
        if(request == null || response == null){
            throw new IllegalArgumentException("request or response must not be null");
        }

        HttpRequestDto dto = get();
        dto.init(request, response);
    }

    /**
     * 清空线程本地变量,必须在处理完用户请求之后调用（一般是在LoginFilter结束时）
     */
    public static void clearThreadLocally() {
        local.remove();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        RemoteAppService remoteAppService = applicationContext.getBean(RemoteAppService.class);
        if(remoteAppService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteAppService(in developer-center)");
        }
        RemoteConsumerService remoteConsumerService = applicationContext.getBean(RemoteConsumerService.class);
        if(remoteAppService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteConsumerService(in consumer-center)");
        }
        RemoteKmsService remoteKmsService = applicationContext.getBean(RemoteKmsService.class);
        if(remoteKmsService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteKmsService(in idmaker-service)");
        }
        oldConsumerEncryptKey = applicationContext.getEnvironment().getProperty("app.consumer.encrypt.key");
        if(oldConsumerEncryptKey == null || oldConsumerEncryptKey.isEmpty()){
            throw new IllegalStateException("property:[app.consumer.encrypt.key] does not exist");
        }
        RequestLocal.remoteAppService = remoteAppService;
        RequestLocal.remoteConsumerService = remoteConsumerService;
        RequestLocal.duibaConsumerCookieClient = new DuibaConsumerCookieClient(remoteKmsService);
    }
}
