package cn.com.duibabiz.component.oss.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duibabiz.component.oss.OssClient;
import cn.com.duibabiz.component.oss.OssProperties;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.net.URLEncoder;
import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.URLUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

/**
 * @author XuJing
 * @since 2020/5/14 10:11 上午
 */
public class OssClientImpl implements OssClient {
    private static final Logger logger = LoggerFactory.getLogger(OssClientImpl.class);


    @Autowired
    private OssProperties ossProperties;

    @Override
    public String upload(String bucketName, String path, File file) throws BizException {
        return upload(bucketName, path, file, null, null, null);
    }

    @Override
    public String upload(String bucketName, String path, String base64) throws BizException {
        return upload(bucketName, path, null, base64, null, null);
    }

    @Override
    public String upload(String bucketName, String path, InputStream inputStream, String fileName) throws BizException {
        if (StringUtils.isBlank(fileName) || inputStream == null) {
            throw new BizException("文件名或者流不能为空");
        }
        String[] split = fileName.split("\\.");
        if (split.length != 2) {
            throw new BizException("文件名格式错误");
        }
        return upload(bucketName, path, null, null, inputStream, fileName);
    }

    @Override
    public InputStream download(String bucketName, String path) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            throw new BizException("bucketName 不能为空");
        }
        if (StringUtils.isBlank(path)) {
            throw new BizException("文件路径 不能为空");
        }
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            String downloadUrl = ossProperties.getDownloadUrl();
            if (downloadUrl.contains("?")) {
                downloadUrl = downloadUrl + "bucketName=%s&objectName=%s";
            } else {
                downloadUrl = downloadUrl + "?bucketName=%s&objectName=%s";
            }
            try {
                String getUrl = String.format(downloadUrl, bucketName, path);
                HttpUtil.download(encode(getUrl), out, false);
            } catch (Exception e) {
                logger.error("下载错误", e);
                throw new BizException("下载错误");
            }
            return new ByteArrayInputStream(out.toByteArray());
        } catch (IOException e) {
            logger.error("系统异常", e);
            throw new BizException("系统异常");
        }
    }

    @Override
    public InputStream download(String url) throws BizException {
        if (StringUtils.isBlank(url)) {
            throw new BizException("url 不能为空");
        }
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            try {
                HttpUtil.download(encode(url), out, false);
            } catch (Exception e) {
                logger.error("系统异常", e);
                throw new BizException("系统异常");
            }
            return new ByteArrayInputStream(out.toByteArray());
        } catch (IOException e) {
            logger.error("系统异常", e);
            throw new BizException("系统异常");
        }
    }

    @Override
    public Boolean fileExist(String bucketName, String path) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            throw new BizException("bucketName 不能为空");
        }
        if (StringUtils.isBlank(path)) {
            throw new BizException("文件路径 不能为空");
        }
        String downloadUrl = ossProperties.getDownloadUrl();
        if (downloadUrl.contains("?")) {
            downloadUrl = downloadUrl + "bucketName=%s&objectName=%s";
        } else {
            downloadUrl = downloadUrl + "?bucketName=%s&objectName=%s";
        }
        try {
            String getUrl = String.format(downloadUrl, bucketName, path);
            final HttpResponse response = HttpRequest.get(encode(getUrl)).executeAsync();
            if (false == response.isOk()) {
                return false;
            }
        } catch (Exception e) {
            logger.error("系统错误", e);
            throw new BizException("系统错误");
        }
        return true;
    }

    public String upload(String bucketName, String path, File file, String base64, InputStream in, String fileName) throws BizException {
        HttpRequest post = HttpUtil.createPost(ossProperties.getUploadUrl());
        Map<String, Object> map = new HashMap<>();
        map.put("bucketName", bucketName);
        map.put("path", path);
        if (file != null) {
            post.form("file", file);
        }
        if (in != null && StringUtils.isNotBlank(fileName)) {
            post.form("file", IoUtil.readBytes(in), fileName);
        }
        if (StringUtils.isNotBlank(base64)) {
            map.put("base64", base64);
        }
        post.form(map);
        post.timeout(5000);
        HttpResponse execute = post.execute();
        try {
            String result = execute.body();
            JSONObject jsonObject = JSON.parseObject(result);
            Boolean success = jsonObject.getBoolean("success");
            if (!success) {
                throw new BizException(jsonObject.getString("message"));
            } else {
                return jsonObject.getString("data");
            }
        } catch (Exception e) {
            logger.error("系统异常", e);
        }
        throw new BizException("系统异常");
    }

    private String encode(String url) {
        URLEncoder encoder = URLEncoder.createDefault();
        encoder.addSafeCharacter('?');
        return encoder.encode(url, Charset.forName(CharsetUtil.UTF_8));
    }
}
