package cn.com.duibabiz.component.oss.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duibabiz.component.oss.OssAppendEntity;
import cn.com.duibabiz.component.oss.OssClient;
import cn.com.duibabiz.component.oss.OssProperties;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * @author XuJing
 * @since 2020/5/14 10:11 上午
 */
public class OssClientImpl implements OssClient {
    private static final Logger logger = LoggerFactory.getLogger(OssClientImpl.class);


    @Autowired
    private OssProperties ossProperties;

    @Resource(name = "ossHttpClient")
    @CanAccessInsideNetwork
    private CloseableHttpClient httpClient;

    @Override
    public String upload(String bucketName, String path, File file) throws BizException {
        return upload(bucketName, path, file, null, null, null);
    }

    @Override
    public String upload(String bucketName, String path, String base64) throws BizException {
        return upload(bucketName, path, null, base64, null, null);
    }

    @Override
    public String upload(String bucketName, String path, InputStream inputStream, String fileName) throws BizException {
        if (StringUtils.isBlank(fileName) || inputStream == null) {
            throw new BizException("文件名或者流不能为空");
        }
        String[] split = fileName.split("\\.");
        if (split.length != 2) {
            throw new BizException("文件名格式错误");
        }
        return upload(bucketName, path, null, null, inputStream, fileName);
    }

    @Override
    public OssAppendEntity appendUpload(String bucketName, String path, InputStream inputStream, String fileName, Long position) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            bucketName = ossProperties.getBucketName();
        }
        if (inputStream == null || position == null || StringUtils.isBlank(fileName)) {
            throw new BizException("必填参数不能为空");
        }
        if (position.equals(0L) && StringUtils.isBlank(path)) {
            throw new BizException("文件路径不能为空");
        }
        String[] split = fileName.split("\\.");
        if (split.length != 2) {
            throw new BizException("文件名格式错误");
        }
        HttpPost httpPost = new HttpPost(ossProperties.getAppendUploadUrl());
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.addPart("bucketName", new StringBody(bucketName, ContentType.create("text/plain", Consts.UTF_8)));
        builder.addPart("position", new StringBody(position.toString(), ContentType.create("text/plain", Consts.UTF_8)));
        builder.addBinaryBody("file", inputStream, ContentType.DEFAULT_BINARY, fileName);
        if (StringUtils.isNotBlank(path)) {
            builder.addPart("path", new StringBody(path, ContentType.create("text/plain", Consts.UTF_8)));
        }
        HttpEntity reqEntity = builder.build();
        httpPost.setEntity(reqEntity);
        try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
            JSONObject jsonObject = JSON.parseObject(EntityUtils.toString(response.getEntity()));
            if (jsonObject.getBoolean("success")) {
                return JSON.parseObject(jsonObject.getJSONObject("data").toJSONString(), OssAppendEntity.class);
            } else {
                throw new BizException(jsonObject.getString("message"));
            }
        } catch (Exception e) {
            logger.error("系统异常 bucketName={},path={},fileName={}", bucketName, path, fileName, e);
        }
        throw new BizException("追加上传失败");
    }

    @Override
    public InputStream download(String bucketName, String path) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            bucketName = ossProperties.getBucketName();
        }
        if (StringUtils.isBlank(path)) {
            throw new BizException("文件路径 不能为空");
        }
        String downloadUrl = ossProperties.getDownloadUrl();
        if (downloadUrl.contains("?")) {
            downloadUrl = downloadUrl + "bucketName=%s&objectName=%s";
        } else {
            downloadUrl = downloadUrl + "?bucketName=%s&objectName=%s";
        }
        String getUrl = String.format(downloadUrl, bucketName, path);
        return httpDownload(getUrl);
    }


    @Override
    public InputStream download(String url) throws BizException {
        if (StringUtils.isBlank(url)) {
            throw new BizException("url 不能为空");
        }
        return httpDownload(url);
    }

    @Override
    public Boolean fileExist(String bucketName, String path) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            bucketName = ossProperties.getBucketName();
        }
        if (StringUtils.isBlank(path)) {
            throw new BizException("文件路径 不能为空");
        }
        String downloadUrl = ossProperties.getDownloadUrl();
        if (downloadUrl.contains("?")) {
            downloadUrl = downloadUrl + "bucketName=%s&objectName=%s";
        } else {
            downloadUrl = downloadUrl + "?bucketName=%s&objectName=%s";
        }
        try {
            String getUrl = String.format(downloadUrl, bucketName, path);
            httpDownload(getUrl);
            return true;
        } catch (BizException e) {
            return false;
        }
    }

    private InputStream httpDownload(String url) throws BizException {
        HttpGet get = new HttpGet(url);
        /**
         * 这句配置，可以让你的请求可以传输中文 能成功
         */
        get.setHeader("Accept-Language", "zh-cn,zh;q=0.5");
        try (CloseableHttpResponse response = httpClient.execute(get)) {
            if (response.getStatusLine().getStatusCode() == 200) {
                //httpClient的流只能读一次 所以做一下转化
                return new ByteArrayInputStream(inputStream2byte(response.getEntity().getContent()));
            }
        } catch (Exception e) {
            logger.error("下载错误 url={}", url, e);
        }
        throw new BizException("下载错误");
    }

    public String upload(String bucketName, String path, File file, String base64, InputStream in, String fileName) throws BizException {
        if (StringUtils.isBlank(bucketName)) {
            bucketName = ossProperties.getBucketName();
        }
        try {
            HttpPost httpPost = new HttpPost(ossProperties.getUploadUrl());
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.addPart("bucketName", new StringBody(bucketName, ContentType.create("text/plain", Consts.UTF_8)));
            if (StringUtils.isNotBlank(path)) {
                builder.addPart("path", new StringBody(path, ContentType.create("text/plain", Consts.UTF_8)));
            }
            if (file != null) {
                builder.addBinaryBody("file", new FileInputStream(file), ContentType.DEFAULT_BINARY, file.getName());
            }
            if (in != null && StringUtils.isNotBlank(fileName)) {
                builder.addBinaryBody("file", in, ContentType.DEFAULT_BINARY, fileName);
            }
            if (StringUtils.isNotBlank(base64)) {
                builder.addPart("base64", new StringBody(base64, ContentType.create("text/plain", Consts.UTF_8)));
            }
            HttpEntity reqEntity = builder.build();
            httpPost.setEntity(reqEntity);
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                if (response.getStatusLine().getStatusCode() != 200) {
                    throw new BizException("系统异常");
                }
                JSONObject jsonObject = JSON.parseObject(EntityUtils.toString(response.getEntity()));
                Boolean success = jsonObject.getBoolean("success");
                if (!success) {
                    throw new BizException(jsonObject.getString("message"));
                } else {
                    return jsonObject.getString("data");
                }
            }
        } catch (Exception e) {
            logger.error("系统异常 bucketName={},path={},fileName={}", bucketName, path, file == null ? fileName : file.getName(), e);
        }

        throw new BizException("系统异常");
    }

    public byte[] inputStream2byte(InputStream inputStream) throws IOException {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        byte[] buff = new byte[100];
        int rc = 0;
        while ((rc = inputStream.read(buff, 0, 100)) > 0) {
            byteArrayOutputStream.write(buff, 0, rc);
        }
        return byteArrayOutputStream.toByteArray();
    }
}
