package cn.com.duiba.biz.tool.duiba.util;


import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 数字金额转中文大写
 * author zhanghuifeng
 * date 2018-11-29-14:04
 */
public class MoneyToCNFormat {
    private static final Pattern AMOUNT_PATTERN =
            Pattern.compile("^(0|[1-9]\\d{0,11})\\.(\\d\\d)$"); // 不考虑分隔符的正确性
    private static final char[] RMB_NUMS = "零壹贰叁肆伍陆柒捌玖".toCharArray();
    private static final String[] UNITS = {"圆", "角", "分", "整"};
    private static final String[] U1 = {"", "拾", "佰", "仟"};
    private static final String[] U2 = {"", "万", "亿"};

    private MoneyToCNFormat(){
        // to do something
    }

    /**
     * 将金额（整数部分等于或少于12位，小数部分2位）转换为中文大写形式.
     * @param amount 金额数字
     * @return       中文大写
     * @throws IllegalArgumentException
     */
    public static String convert(String amount) {
        // 去掉分隔符
        amount = amount.replace(",", "");

        // 验证金额正确性
        if (amount.equals("0.00")) {
            return "零圆";
        }
        Matcher matcher = AMOUNT_PATTERN.matcher(amount);
        if (! matcher.find()) {
            return amount;
        }

        String integer  = matcher.group(1); // 整数部分
        String fraction = matcher.group(2); // 小数部分

        String result = "";
        if (! integer.equals("0")) {
            result += integer2rmb(integer) + UNITS[0]; // 整数部分
        }
        if (fraction.equals("00")) {
            result += UNITS[3]; // 添加[整]
        } else if (fraction.startsWith("0") && integer.equals("0")) {
            result += fraction2rmb(fraction).substring(1); // 去掉分前面的[零]
        } else {
            result += fraction2rmb(fraction); // 小数部分
        }

        return result;
    }

    // 将金额小数部分转换为中文大写
    private static String fraction2rmb(String fraction) {
        char jiao = fraction.charAt(0); // 角
        char fen  = fraction.charAt(1); // 分
        return (RMB_NUMS[jiao - '0'] + (jiao > '0' ? UNITS[1] : ""))
                + (fen > '0' ? RMB_NUMS[fen - '0'] + UNITS[2] : "");
    }

    // 将金额整数部分转换为中文大写
    private static String integer2rmb(String integer) {
        StringBuilder buffer = new StringBuilder();
        // 从个位数开始转换
        int i;
        int j;
        int size = integer.length();
        for (i = size - 1, j = 0; i >= 0; i--, j++) {
            dealSingle(integer,j,i,buffer,size);
        }
        return buffer.reverse().toString();
    }

    private static void dealSingle(String integer, int j, int i, StringBuilder buffer, int size){
        char n = integer.charAt(i);
        if (n == '0') {
            deal0(integer, j, i, buffer);
        } else {
            if (j % 4 == 0) {
                buffer.append(U2[j / 4]);     // 插入[万]或者[亿]
            }
            buffer.append(U1[j % 4]);         // 插入[拾]、[佰]或[仟]
            // 壹拾X元 -> 拾X元，壹是没必要显示
            if (!(size == 2 && i == 0 && n == '1')) {
                buffer.append(RMB_NUMS[n - '0']); // 插入数字
            }
        }
    }

    private static void deal0(String integer, int j, int i, StringBuilder buffer){
        // 当n是0且n的右边一位不是0时，插入[零]
        if (i < integer.length() - 1 && integer.charAt(i + 1) != '0') {
            buffer.append(RMB_NUMS[0]);
        }
        // 插入[万]或者[亿]
        if (j % 4 == 0 && (i > 0 && integer.charAt(i - 1) != '0'
                || i > 1 && integer.charAt(i - 2) != '0'
                || i > 2 && integer.charAt(i - 3) != '0')) {
            buffer.append(U2[j / 4]);
        }
    }
}