package cn.com.duiba.biz.tool.duiba.client;

import cn.com.duiba.biz.tool.duiba.dto.ConsumerCookieDto;
import cn.com.duiba.consumer.center.api.dto.ConsumerDto;
import cn.com.duiba.idmaker.service.api.client.kms.KmsClient;
import cn.com.duiba.idmaker.service.api.dto.kms.TimeBasedRollingKeyDto;
import cn.com.duiba.idmaker.service.api.enums.kms.KeyTypeEnums;
import cn.com.duiba.idmaker.service.api.enums.kms.KeyUseTypeEnums;
import cn.com.duiba.idmaker.service.api.remoteservice.kms.RemoteKmsService;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 兑吧用户客户端，用于从request cookie获取用户信息或者注入用户信息到cookie中<br/>
 *
 * Created by wenqi.huang on 2017/5/5.
 */
public class DuibaConsumerCookieClient {

    protected static final String CONSUMER_WDATA4_COOKIE = "wdata4";
    private static final String COMMERCIAL_CONSUMER_WDATA3_COOKIE = "wdata3";
    private static final String CONSUMER_YJQ_INFO_COOKIE = "yjq_info";
    protected static final String LOGIN_TIME_COOKIE = "w_ts";
    protected static final String YJQ_LOGIN_TIME_COOKIE = "yjq_ts";
    protected static final String TOKEN_ID_COOKIE = "tokenId";
    private static final String X_HOST = "xhost";
    private static final int PERIOD_FOR_EVER = 10*365*24*60*60;

    private final KmsClient kmsClient;

    public DuibaConsumerCookieClient(RemoteKmsService remoteKmsService){
        this.kmsClient = new KmsClient(remoteKmsService);
    }

    /**
     * 从request cookie中获取用户信息，未登录或超过24小时则返回null
     * @param request
     * @return
     */
    public ConsumerCookieDto getConsumerCookieDto(HttpServletRequest request){
        String wdata4 = RequestTool.getCookie(request, CONSUMER_WDATA4_COOKIE);
        if(wdata4 == null || wdata4.isEmpty()){
            return null;
        }
        long ts =  NumberUtils.parseLong(RequestTool.getCookie(request, LOGIN_TIME_COOKIE), -1L);
        if(ts == -1L || !isValidTime(ts)){
            return null;
        }
        TimeBasedRollingKeyDto tbrk = getTimeBasedRollingKey(ts);
        String userJson = kmsClient.decrypt(wdata4, tbrk);//如果ts是伪造的，则解密时可能会抛出异常，需要处理
        ConsumerCookieDto c = JSONObject.parseObject(userJson, ConsumerCookieDto.class);
        if(!isValidTime(c.getTime())){
            return null;
        }
        return c;
    }

    /**
     * 从request cookie中获取用户信息，未登录或超过24小时则返回null
     *
     * @param request
     * @return
     */
    public ConsumerCookieDto getConsumerCookieDtoForYjq(HttpServletRequest request) {
        String yjqInfo = RequestTool.getCookie(request, CONSUMER_YJQ_INFO_COOKIE);
        if (yjqInfo == null || yjqInfo.isEmpty()) {
            return null;
        }
        long ts = NumberUtils.parseLong(RequestTool.getCookie(request, YJQ_LOGIN_TIME_COOKIE), -1L);
        if (ts == -1L || !isValidTime(ts)) {
            return null;
        }
        TimeBasedRollingKeyDto tbrk = getTimeBasedRollingKey(ts);
        String userJson = kmsClient.decrypt(yjqInfo, tbrk);//如果ts是伪造的，则解密时可能会抛出异常，需要处理
        ConsumerCookieDto c = JSONObject.parseObject(userJson, ConsumerCookieDto.class);
        if (!isValidTime(c.getTime())) {
            return null;
        }
        return c;
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac三个cookie)
     * @param consumer
     * @param response
     * @param cookieDomain cookie需要设置的domain
     * @param isFromCommercial 是否来自商业流量, 如果是商业流量，则用户信息会放入wdata3中，永久有效；否则用户信息放入wdata4中，24小时有效
     * @return 设置到cookie中的用户信息
     */
    public ConsumerCookieDto injectConsumerInfoIntoCookie(ConsumerDto consumer, HttpServletRequest request, HttpServletResponse response, String cookieDomain, String oldConsumerEncryptKey, boolean isFromCommercial){
        long loginTime = System.currentTimeMillis();
        ConsumerCookieDto cookieDto = makeConsumerCookieDto(consumer, loginTime);
        if(isFromCommercial){
            cookieDto.setForEver(true);
        }
        String userJson = JSONObject.toJSONString(cookieDto);

        String userCookieVal;
        if(isFromCommercial){//商业流量使用固定密钥加密，因为cookie是永久有效的
            userCookieVal = BlowfishUtils.encryptBlowfish(userJson, oldConsumerEncryptKey);
        }else {
            TimeBasedRollingKeyDto tbrk = getTimeBasedRollingKey(loginTime);
            userCookieVal = kmsClient.encrypt(userJson, tbrk);
        }
        String reqDomain = getRequestDomain(request,cookieDomain);
        if(reqDomain != null){
            Cookie reqUserCookie = new Cookie(isFromCommercial ? COMMERCIAL_CONSUMER_WDATA3_COOKIE : CONSUMER_WDATA4_COOKIE, userCookieVal);
            reqUserCookie.setHttpOnly(true);//防止被js获得
            reqUserCookie.setDomain(reqDomain);
            reqUserCookie.setPath("/");
            if(isFromCommercial){
                reqUserCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(reqUserCookie);
        } else {
            Cookie userCookie = new Cookie(isFromCommercial ? COMMERCIAL_CONSUMER_WDATA3_COOKIE : CONSUMER_WDATA4_COOKIE, userCookieVal);
            userCookie.setHttpOnly(true);//防止被js获得
            userCookie.setDomain(cookieDomain);
            userCookie.setPath("/");
            if (isFromCommercial) {
                userCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(userCookie);
        }

        String loginTimeStr = String.valueOf(loginTime);
        if(reqDomain != null){
            Cookie reqTsCookie = new Cookie(LOGIN_TIME_COOKIE, loginTimeStr);
            reqTsCookie.setHttpOnly(true);//防止被js获得
            reqTsCookie.setDomain(reqDomain);
            reqTsCookie.setPath("/");
            if(isFromCommercial){
                reqTsCookie.setValue("");
                reqTsCookie.setMaxAge(0);//商业流量使用了wdata3，所以无需保存时间戳，删除该cookie
            }
            response.addCookie(reqTsCookie);
        } else {
            Cookie tsCookie = new Cookie(LOGIN_TIME_COOKIE, loginTimeStr);
            tsCookie.setHttpOnly(true);//防止被js获得
            tsCookie.setDomain(cookieDomain);
            tsCookie.setPath("/");
            if (isFromCommercial) {
                tsCookie.setValue("");
                tsCookie.setMaxAge(0);//商业流量使用了wdata3，所以无需保存时间戳，删除该cookie
            }
            response.addCookie(tsCookie);
        }

        //植入appId和用户id，nginx可以获取该字段,应用不应该使用这个cookie，很容易被用户篡改
        JSONObject aidCidObject = new JSONObject();
        aidCidObject.put("aid", consumer.getAppId());
        aidCidObject.put("cid", consumer.getId());
        String aidCidValue = SecurityUtils.encode2StringByBase64(aidCidObject.toJSONString().getBytes());
        if(reqDomain != null){
            Cookie reqAcCookie = new Cookie("_ac", aidCidValue);
            reqAcCookie.setHttpOnly(true);//防止被js获得
            reqAcCookie.setDomain(reqDomain);
            reqAcCookie.setPath("/");
            if(isFromCommercial){
                reqAcCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(reqAcCookie);
        } else {
            Cookie acCookie = new Cookie("_ac", aidCidValue);
            acCookie.setHttpOnly(true);//防止被js获得
            acCookie.setDomain(cookieDomain);
            acCookie.setPath("/");
            if (isFromCommercial) {
                acCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(acCookie);
        }

        String tokenId = SecurityUtils.encode2StringByMd5(userCookieVal);
        if(reqDomain != null){
            Cookie reqTokenIdCookie = new Cookie(TOKEN_ID_COOKIE, tokenId);
            reqTokenIdCookie.setDomain(reqDomain);
            reqTokenIdCookie.setPath("/");
            if(isFromCommercial){
                reqTokenIdCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(reqTokenIdCookie);
        } else {
            Cookie tokenIdCookie = new Cookie(TOKEN_ID_COOKIE, tokenId);
            tokenIdCookie.setDomain(cookieDomain);
            tokenIdCookie.setPath("/");
            if (isFromCommercial) {
                tokenIdCookie.setMaxAge(PERIOD_FOR_EVER);//商业流量的cookie永久有效
            }
            response.addCookie(tokenIdCookie);
        }

        request.setAttribute("c_tokenId", tokenId);

        if(!isFromCommercial) {
            if(reqDomain != null) {
                injectOldWdata3Cookie(userJson, response, reqDomain, oldConsumerEncryptKey);
            } else {
                injectOldWdata3Cookie(userJson, response, cookieDomain, oldConsumerEncryptKey);
            }
        }

        return cookieDto;
    }

    /**
     * 获取本次请求的域名,如果和配置的域名不一致才会返回当前请求的域名,否则返回null
     * 优先会获取x-host这个head
     * @param request       请求对象
     * @param cookieDomain  配置的根域名
     * @return
     */
    public String getRequestDomain(HttpServletRequest request,String cookieDomain){
        String domainName = request.getServerName();
        if (StringUtils.isNotBlank(request.getHeader(X_HOST))) {
            return request.getHeader(X_HOST);
        }
        if(domainName == null || "".equals(domainName)){
            return null;
        }
        if(domainName.contains(cookieDomain)){
            return null;
        }
        return domainName;
    }


    //注入老的wdata3 cookie。等wdata4稳定以后删除。
    @Deprecated
    private void injectOldWdata3Cookie(String userJson, HttpServletResponse response, String cookieDomain, String oldConsumerEncryptKey){
        String wdata3 = BlowfishUtils.encryptBlowfish(userJson, oldConsumerEncryptKey);
        Cookie wdata3Cookie = new Cookie("wdata3", wdata3);
        wdata3Cookie.setDomain(cookieDomain);
        wdata3Cookie.setPath("/");
        response.addCookie(wdata3Cookie);
    }

    /**
     * 登录时间在24小时内为有效
     * @param loginTime
     * @return
     */
    private boolean isValidTime(long loginTime){
        long now = System.currentTimeMillis();
        return loginTime > now - 86400000 && loginTime < now + 300000;
    }

    private TimeBasedRollingKeyDto getTimeBasedRollingKey(long ts){
        return kmsClient.getCachedTimeBasedRollingKey(KeyUseTypeEnums.COOKIE, KeyTypeEnums.AES_128, ts);
    }

    private ConsumerCookieDto makeConsumerCookieDto(ConsumerDto consumerDto, long loginTime){
        if(consumerDto.getAppId() == null){
            throw new IllegalArgumentException("consumer's appId must not be null");
        }
        if(consumerDto.getId() == null){
            throw new IllegalArgumentException("consumer's id must not be null");
        }
        if(consumerDto.getPartnerUserId() == null){
            throw new IllegalArgumentException("consumer's partnerUserId must not be null");
        }
        ConsumerCookieDto cookieDto = new ConsumerCookieDto();
        cookieDto.setAppId(consumerDto.getAppId());
        cookieDto.setCid(consumerDto.getId());
        cookieDto.setPartnerUserId(consumerDto.getPartnerUserId());
        cookieDto.setTime(loginTime);

        return cookieDto;
    }

    public ConsumerCookieDto injectConsumerInfoIntoCookieForYjq(ConsumerDto consumer, HttpServletRequest request, HttpServletResponse response, String cookieDomain, String oldConsumerEncryptKey) {
        long loginTime = System.currentTimeMillis();
        ConsumerCookieDto cookieDto = makeConsumerCookieDto(consumer, loginTime);
        String userJson = JSONObject.toJSONString(cookieDto);
        String userCookieVal;
        TimeBasedRollingKeyDto tbrk = getTimeBasedRollingKey(loginTime);
        userCookieVal = kmsClient.encrypt(userJson, tbrk);
        String reqDomain = getRequestDomain(request, cookieDomain);
        if (reqDomain != null) {
            Cookie reqUserCookie = new Cookie(CONSUMER_YJQ_INFO_COOKIE, userCookieVal);
            reqUserCookie.setHttpOnly(true);//防止被js获得
            reqUserCookie.setDomain(reqDomain);
            reqUserCookie.setPath("/");
            response.addCookie(reqUserCookie);
        } else {
            Cookie userCookie = new Cookie(CONSUMER_YJQ_INFO_COOKIE, userCookieVal);
            userCookie.setHttpOnly(true);//防止被js获得
            userCookie.setDomain(cookieDomain);
            userCookie.setPath("/");
            response.addCookie(userCookie);
        }

        String loginTimeStr = String.valueOf(loginTime);
        if (reqDomain != null) {
            Cookie reqTsCookie = new Cookie(YJQ_LOGIN_TIME_COOKIE, loginTimeStr);
            reqTsCookie.setHttpOnly(true);//防止被js获得
            reqTsCookie.setDomain(reqDomain);
            reqTsCookie.setPath("/");
            response.addCookie(reqTsCookie);
        } else {
            Cookie tsCookie = new Cookie(YJQ_LOGIN_TIME_COOKIE, loginTimeStr);
            tsCookie.setHttpOnly(true);//防止被js获得
            tsCookie.setDomain(cookieDomain);
            tsCookie.setPath("/");
            response.addCookie(tsCookie);
        }

        return cookieDto;
    }
}
