package cn.com.duibabiz.component.token.impl;

import cn.com.duiba.biz.tool.duiba.util.DuibaTokenUtil;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.JavaScriptUtil;
import cn.com.duibabiz.component.token.TokenService;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.util.Assert;

import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

/**
 * Created by HePeng on 2018/12/11 16:06.
 */
public class TokenServiceImpl implements TokenService {
    private final AdvancedCacheClient advancedCacheClient;
    private final String multiDummyTokenJs;

    /**
     * 线上域名标志
     */
    private static final String ONLINE_DOMAIN_MARK = "duiba.com.cn";
    /**
     * 表单token缓存key前缀
     */
    private static final int MS_FORM_CONSUMER_TOKEN = 102;
    /**
     * 表单token缓存分钟数
     */
    private static final int SECRET_TOKEN_CACHE_MINUTES = 5;
    /**
     * 混淆token数量
     */
    private static final int OBFUSCATE_TOKEN_COUNT = 10;
    /**
     * 测试token
     */
    private static final String TEST_TOKEN = "yrJRr7Cddp2YeQd";

    public TokenServiceImpl(AdvancedCacheClient advancedCacheClient, String multiDummyTokenJs) {
        this.advancedCacheClient = advancedCacheClient;
        this.multiDummyTokenJs = multiDummyTokenJs;
    }

    @Override
    public String getAndCacheSecretToken(Long consumerId) {
        Assert.notNull(consumerId, "用户id不能为空");
        return getAndCacheSecretToken(getSecretTokenCacheKey(consumerId));
    }

    @Override
    public String getAndCacheSecretToken(String customizedCacheKey) {
        Assert.notNull(customizedCacheKey, "定制缓存key不能为空");
        return advancedCacheClient.getWithCacheLoader(customizedCacheKey
                , SECRET_TOKEN_CACHE_MINUTES, TimeUnit.MINUTES, false, DuibaTokenUtil::genToken);
    }

    @Override
    public String getConfusedToken(Long consumerId, String referer) {
        return getConfusedToken(consumerId, referer,
                consumerId != null ? getSecretTokenCacheKey(consumerId) : null);
    }

    @Override
    public String getConfusedToken(Long consumerId, String referer, String customizedCacheKey) {
        String key;
        String token;
        if(consumerId == null || !passByReferer(referer)) { // 校验不通过
            key = DuibaTokenUtil.genDummyTokenKey();
            token = DuibaTokenUtil.genToken();
        } else {
            key = DuibaTokenUtil.getSecretTokenKey(consumerId);
            token = getAndCacheSecretToken(customizedCacheKey);
        }
        return JavaScriptUtil.obfuscateScript(obfuscateToken(OBFUSCATE_TOKEN_COUNT, key, token));
    }

    @Override
    public boolean checkAndInvalidConsumerToken(Long consumerId, String token) {
        if (consumerId == null || StringUtils.isBlank(token)) {
            return false;
        }
        return checkAndInvalidConsumerToken(getSecretTokenCacheKey(consumerId), token);
    }

    @Override
    public boolean checkAndInvalidConsumerToken(String customizedCacheKey, String token) {
        if(StringUtils.isBlank(customizedCacheKey) || StringUtils.isBlank(token)) {
            return false;
        }
        // 测试token，直接通过
        if (TEST_TOKEN.equals(token)) {
            return true;
        }
        // 测试环境，直接通过
        if(SpringEnvironmentUtils.isTestEnv()) {
            return true;
        }
        // 获取缓存中的token
        String cacheToken = advancedCacheClient.get(customizedCacheKey);
        if (StringUtils.isBlank(cacheToken)) {
            return false;
        }
        // 移除缓存token并比对
        advancedCacheClient.remove(customizedCacheKey);
        return cacheToken.equals(token);
    }

    /**
     * 混淆token
     * @param count
     * @param key
     * @param token
     * @return
     */
    private String obfuscateToken(int count, String key, String token) {
        StringBuilder sb = new StringBuilder();
        int keyRandomIndex = RandomUtils.nextInt(count);
        IntStream.range(0, count).forEach(i -> {
            sb.append("window['")
                    .append(DuibaTokenUtil.genDummyTokenKey())
                    .append("']=\"")
                    .append(DuibaTokenUtil.genToken())
                    .append("\";");
            if (keyRandomIndex == i) {
                sb.append("window['")
                        .append(key)
                        .append("']=\"")
                        .append(token)
                        .append("\";");
            }
        });
        // 批量生成混淆token的js
        if (StringUtils.isNotBlank(multiDummyTokenJs)) {
            sb.append(multiDummyTokenJs);
        }
        return sb.toString();
    }

    /**
     * 获取表单token的缓存key
     * @param consumerId
     * @return
     */
    private String getSecretTokenCacheKey(Long consumerId) {
        return MS_FORM_CONSUMER_TOKEN + "-" + consumerId;
    }

    /**
     * 检查referer是否通过
     * @param referer
     * @return
     */
    private static boolean passByReferer(String referer) {
        return StringUtils.isNotBlank(referer)
                && (referer.contains(ONLINE_DOMAIN_MARK) || !SpringEnvironmentUtils.isProdEnv());
    }
}
