package cn.com.duiba.biz.tool.duiba.client;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * @author zhangruofan
 * @date 2018/04/14
 */
public final class DateUtils {
    private static final Logger log = LoggerFactory.getLogger(DateUtils.class);

    private static final String DAY = "yyyy-MM-dd";
    private static final String MONTH = "yyyy-MM";

    /**
     * Private constructor since this is a utility class.
     */
    private DateUtils() {

    }
    /**
     * 获取当月最后一天
     * @return LocalDate
     */
    public static LocalDate getLastDayOfMonth(LocalDate localDate) {
        return localDate.withDayOfMonth(localDate.lengthOfMonth());
    }

    /**
     * 获取当周最后一天(周日作为最后一天)
     * @return LocalDate
     */
    public static LocalDate getLastDayOfWeek(LocalDate localDate) {
        return localDate.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY));
    }

    /**
     *
     * getDayDate:(这里用一句话描述这个方法的作用). <br/>
     * method : 19、根据"yyyy-MM-dd"形式的字符串 获取Date
     * @param dateStr
     * @return
     * @since JDK 1.6
     */
    public static Date getDayDate(String dateStr) {
        return getDate(dateStr, new SimpleDateFormat(DAY));
    }

    private static Date getDate(String dateStr, SimpleDateFormat format) {
        if ("".equals(dateStr) || dateStr == null)
            return null;
        try {
            return format.parse(dateStr);
        } catch (ParseException e) {
            log.error("format yyyy-MM-dd HH:mm:ss error:", e);
        }
        return null;
    }

    /**
     * 获取到指定日期24：00：00还剩下多少秒
     * @return
     */
    public static long timeDifferenceSeconds(String dataStr,String dateFormat){
        Calendar curDate = Calendar.getInstance();
        Calendar toDate = Calendar.getInstance();
        try {
            toDate.setTime(new SimpleDateFormat(dateFormat).parse(dataStr));
        } catch (ParseException e) {
            return 0;
        }
        Calendar toDateNew = new GregorianCalendar(toDate
                .get(Calendar.YEAR), toDate.get(Calendar.MONTH), toDate
                .get(Calendar.DATE) + 1, 0, 0, 0);
        return (toDateNew.getTimeInMillis() - curDate .getTimeInMillis()) / 1000;

    }

    /**
     * 获取当前时间到明天0点的秒数（至少返回1，以防redis用0作为超时时间报错）
     */
    public static int getToTomorrowSeconds(){
        Calendar curDate = Calendar.getInstance();
        Calendar tommorowDate = new GregorianCalendar(curDate.get(Calendar.YEAR), curDate.get(Calendar.MONTH),
                curDate.get(Calendar.DATE) + 1, 0, 0, 0);
        return Math.max((int) ((tommorowDate.getTimeInMillis() - curDate.getTimeInMillis()) / 1000), 1);
    }

    /**
     * day天之后的日期
     * @param day
     * @return
     */
    public static String nextDay(int day) {
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.add(Calendar.DAY_OF_MONTH, day);// 今天+day天

        Date next = c.getTime();
        return new SimpleDateFormat(DAY).format(next);
    }

    /**
     * 本周最后一天
     * @return
     */
    public static String getLastOfWeek(){
        Calendar cal = Calendar.getInstance();
        int d;
        if (cal.get(Calendar.DAY_OF_WEEK) == 1) {
            d = 0;
        } else {
            d = 8 - cal.get(Calendar.DAY_OF_WEEK);
        }
        cal.add(Calendar.DAY_OF_WEEK, d);
        // 所在周结束日期
        return new SimpleDateFormat(DAY).format(cal.getTime());
    }

    /**
     * 获取本月最后一天
     * @return
     */
    public static String getLastOfMonth(){
        //获取当前月最后一天
        Calendar ca = Calendar.getInstance();
        ca.set(Calendar.DAY_OF_MONTH, ca.getActualMaximum(Calendar.DAY_OF_MONTH));
        return new SimpleDateFormat(DAY).format(ca.getTime());
    }
    
    /**
     * 获取本月第一天
     * 
     * @return
     */
    public static Date getFistDayOfMonth() {
        //获取当前月最后一天
        Calendar ca = Calendar.getInstance();
        ca.add(Calendar.MONTH, 0);
        ca.set(Calendar.DAY_OF_MONTH, 1);
        ca.set(Calendar.HOUR_OF_DAY,0);
        ca.set(Calendar.MINUTE,0);
        ca.set(Calendar.SECOND,0);
        return ca.getTime();
    }
    
    /**
     * 获取第二个时间和第一个时间  月份差
     * 
     * @return
     */
    public static int timeDifferenceMonth(Date timeStar,Date timeEnd){
        //获取当前月最后一天
        Calendar ca = Calendar.getInstance();
        ca.setTime(timeStar);
        int yearStat = ca.get(Calendar.YEAR);
        int monStat = ca.get(Calendar.MONTH) + 1;
        ca.setTime(timeEnd);
        int monEnd = ca.get(Calendar.MONTH) + 1;
        int yearEnd = ca.get(Calendar.YEAR);
        return (yearEnd-yearStat)*12+monEnd -monStat;
    }

    /**
    *
    * getDayDate:(这里用一句话描述这个方法的作用). <br/>
    * method : 19、根据"yyyy-MM-dd"形式的字符串 获取Date 前一个月
    * @param time
    * @return
    * @since JDK 1.6
    */
    public static String getYYYYMM(Date time) {
    	SimpleDateFormat sf =new SimpleDateFormat(MONTH);
        Calendar ca = Calendar.getInstance();
        ca.setTime(time);
        ca.add(Calendar.MONTH, -1);
    	return sf.format(ca.getTime());
    }
    
    /**
    *
    * getDayDate:(这里用一句话描述这个方法的作用). <br/>
    * method : 19、根据"yyyy-MM-dd"形式的字符串 获取Date 前一个月
    * @param time
    * @return
    * @since JDK 1.6
    */
    public static int getWeekOfYear(Date time) {
        Calendar ca = Calendar.getInstance();
        ca.setTime(time);
    	return ca.get(Calendar.WEEK_OF_YEAR);
    }

	/**
	 * 
	 * getDayStr:(这里用一句话描述这个方法的作用). <br/>
	 * method : 13、根据Date获取"yyyy-MM-dd"字符串
	 * @param date
	 * @return
	 * @since JDK 1.6
	 */
	public static String getDayStr(Date date) {
		return getStr(date, new SimpleDateFormat(DAY));
	}
	
	private static String getStr(Date date, SimpleDateFormat format) {
		if (date == null) {
			return "";
		}
		return format.format(date);
	}
}
