package cn.com.duiba.boot.utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.net.util.IPAddressUtil;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.regex.Pattern;

/**
 * Created by wenqi.huang on 2017/6/19.
 */
public class NetUtils {

    private NetUtils(){}

    private static final Logger log = LoggerFactory.getLogger(NetUtils.class);

    public static final String ANYHOST = "0.0.0.0";//NOSONAR
    private static final Pattern IP_PATTERN = Pattern.compile("\\d{1,3}(\\.\\d{1,3}){3,5}$");

    private static volatile String localIP = null;

    /**
     * 获取本机内网IP(结果会缓存)
     *
     * @return
     */
    public static String getLocalIp() {
        if (localIP != null) {
            return localIP;
        }
        try {
            InetAddress localAddress = InetAddress.getLocalHost();
            if (isValidLocalAddress(localAddress)) {
                localIP = localAddress.getHostAddress();
                return localIP;
            }
        } catch (Throwable e) {
            log.warn("Failed to retriving ip address, " + e.getMessage(), e);
        }
        Enumeration netInterfaces;
        try {
            netInterfaces = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException e) {
            log.error("getLocalIp error", e);
            return null;
        }
        while (netInterfaces.hasMoreElements()) {
            NetworkInterface ni = (NetworkInterface) netInterfaces.nextElement();
            Enumeration address = ni.getInetAddresses();
            while (address.hasMoreElements()) {
                InetAddress ip = (InetAddress) address.nextElement();
                if (ip.isSiteLocalAddress() && !ip.isLoopbackAddress() && ip.getHostAddress().indexOf(':') == -1) {//内网IP
                    localIP = ip.getHostAddress();
                    return localIP;
                }
            }
        }
        return localIP;
    }

    /**
     * 判断是否合法的本机内网ip
     * @return
     */
    private static boolean isValidLocalAddress(InetAddress address){
        if (address == null || address.isLoopbackAddress() || !address.isSiteLocalAddress()) {
            return false;
        }
        if(address instanceof Inet6Address){
            return false;
        }
        String name = address.getHostAddress();
        return (name != null
                && ! ANYHOST.equals(name)
                && IP_PATTERN.matcher(name).matches());
    }

    /**
     * 是否是内网ip
     * 以下范围的属于内网IP：<br/>
     *       10.0.0.0~10.255.255.255（A类）<br/>
     *       172.16.0.0~172.31.255.255（B类）<br/>
     *       192.168.0.0~192.168.255.255（C类）<br/>
     *
     * @param ip
     * @return
     */
    public static boolean isLanIp(String ip) {
        byte[] addr = IPAddressUtil.textToNumericFormatV4(ip);
        return isLanIp(addr);
    }

    private static boolean isLanIp(byte[] addr) {
        if(addr == null || addr.length < 2){
            return false;
        }
        final byte b0 = addr[0];
        final byte b1 = addr[1];
        //10.x.x.x/8
        final byte SECTION_1 = 0x0A;
        //172.16.x.x/12
        final byte SECTION_2 = (byte) 0xAC;
        final byte SECTION_3 = (byte) 0x10;
        final byte SECTION_4 = (byte) 0x1F;
        //192.168.x.x/16
        final byte SECTION_5 = (byte) 0xC0;
        final byte SECTION_6 = (byte) 0xA8;
        if(b0 == SECTION_1){
            return true;
        }else if(b0 == SECTION_2 && b1 >= SECTION_3 && b1 <= SECTION_4){
            return true;
        }else if(b0 == SECTION_5 && b1 == SECTION_6){
            return true;
        }

        return false;
    }
}
