package cn.com.duiba.boot.utils;

import com.google.common.base.Throwables;
import org.apache.commons.collections.CollectionUtils;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import java.io.ByteArrayInputStream;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.*;

/**
 * SSL证书工具类，可以用于解析pem/cert等格式证书中的域名、过期时间等信息
 * <br/>
 */
public class X509CertificateUtils {
    private X509CertificateUtils(){
    }

    /**
     * 获取指定SSL证书对应的所有hostname列表，可能包含ip、或者通配符域名.
     * <br/>
     * 算法思想来源：https://stackoverflow.com/questions/17943286/get-list-of-all-hostnames-matching-certificate/17948270#17948270
     * @param bs
     * @return
     */
    public static List<String> getHostnames(byte[] bs){
        try{
            CertificateFactory certificateFactory = CertificateFactory
                    .getInstance("X.509");
            X509Certificate x509certificate = (X509Certificate) certificateFactory
                    .generateCertificate(new ByteArrayInputStream(bs));

            //从CN中获取hostname
            String dn = x509certificate.getSubjectX500Principal().getName();
            LdapName ldapDN = new LdapName(dn);
            String hostnameFromCN = null;
            for(Rdn rdn: ldapDN.getRdns()) {
                if("CN".equals(rdn.getType())){
                    hostnameFromCN = (String)rdn.getValue();
                    break;
                }
            }

            Set<String> hostnames = new HashSet<>();

            if(hostnameFromCN != null) {
                hostnames.add(hostnameFromCN);
            }

            //从SAN信息中获取hostname
            Collection<List<?>> sanList = x509certificate.getSubjectAlternativeNames();
            if(!CollectionUtils.isEmpty(sanList)) {
                for (List<?> obj : sanList) {
                    List<?> innerList = obj;
                    int type = (Integer) innerList.get(0);
                    String hostname = (String) innerList.get(1);
                    if (type == 2 || type == 7) {//2表示dns、7表示ip
                        hostnames.add(hostname);
                    }
                }
            }

            return new ArrayList<>(hostnames);
        } catch (CertificateException | InvalidNameException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获得ssl证书的截止日期
     * @param bs
     * @return
     */
    public static Date getEndDate(byte[] bs){
        try {
            CertificateFactory certificateFactory = CertificateFactory
                    .getInstance("X.509");
            X509Certificate x509certificate = (X509Certificate) certificateFactory
                    .generateCertificate(new ByteArrayInputStream(bs));

            return x509certificate.getNotAfter();
        } catch (CertificateException e) {
            throw new RuntimeException(e);
        }
    }
}
