package cn.com.duiba.boot.perftest;

import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpCookie;
import org.springframework.http.server.reactive.ServerHttpRequest;
/**
 * 压测工具类,reactive版，用于判断是否压测请求，获取压测场景ID、判断是否独立容器集群压测
 */
public final class ReactivePerfTestUtils {
    private ReactivePerfTestUtils(){}

    /**
     * 检测到包含_duibaPerf=1参数的url(或cookie中有_duibaPerf=1,或header中包含_duibaPerf=1)则认为是压测请求
     * @param request
     * @return
     */
    public static boolean isPerfTestRequest(ServerHttpRequest request){
        boolean isTestMode = false;
        String perfTestHeader = request.getHeaders().getFirst(PerfTestUtils.PERF_TEST_KEY);
        if(perfTestHeader != null && ("1".equals(perfTestHeader) || "true".equals(perfTestHeader))){
            isTestMode = true;
        }

        if(!isTestMode && "true".equals(request.getHeaders().getFirst(PerfTestUtils.IS_PERF_TEST_MODE_HEADER_KEY))){
            //feign 发起的http压测请求头中会带有isPerfTestMode=true
            isTestMode = true;
        }

        if (!isTestMode) {
            HttpCookie cookie = request.getCookies().getFirst(PerfTestUtils.PERF_TEST_KEY);
            if(cookie != null && ("1".equals(cookie.getValue()) || "true".equals(cookie.getValue()))) {
                isTestMode = true;
            }
        }

        if(!isTestMode) {
            isTestMode = StringUtils.contains(request.getURI().getQuery(), PerfTestUtils.PERF_TEST_KEY);//为了提高处理效率，只要query中包含_duibaPerf就认为是压测请求
        }

        return isTestMode;
    }

    /**
     * 从request中获取压测场景id，压测场景ID可以在以下任一地方：
     * <br/>
     * 参数的url中带有 _duibaPerfSceneId=**
     * <br/>
     * cookie中有 _duibaPerfSceneId=**
     * <br/>
     * header中包含 _duibaPerfSceneId=**
     *
     * @param request
     * @return 压测场景ID，可能为null (如果没有压测标记，无论有没有压测场景ID，都返回null)
     */
    public static String getPerfTestSceneId(ServerHttpRequest request){
        if(!isPerfTestRequest(request)){//如果没有压测标记，无论有没有压测场景ID，都返回null
            return null;
        }

        String sceneId = request.getHeaders().getFirst(PerfTestUtils.PERF_TEST_SCENE_ID_KEY);

        if(StringUtils.isBlank(sceneId)){
            HttpCookie cookie = request.getCookies().getFirst(PerfTestUtils.PERF_TEST_SCENE_ID_KEY);
            if(cookie != null) {
                sceneId = cookie.getValue();
            }
        }

        if(StringUtils.isBlank(sceneId)){
            sceneId = request.getQueryParams().getFirst(PerfTestUtils.PERF_TEST_SCENE_ID_KEY);
        }

        return sceneId;
    }

    /**
     * 从 request 中获取压测集群标识，压测集群标识可以再以下任一地方
     * <br/>
     * 参数的url中带有 _duibaPerfTestCluster= 1 or 0
     * <br/>
     * cookie中有 _duibaPerfTestCluster= 1 or 0
     * <br/>
     * header中包含 _duibaPerfTestCluster= 1 or 0
     * @param request
     * @return 当且仅当 _duibaPerfTestCluster = 1 or true 的情况下，返回true，其余时候都是 false
     */
    public static boolean isPerfTestCluster(ServerHttpRequest request) {
        if (!isPerfTestRequest(request)) { // 如果没有压测标记，无论有没有压测集群标识，都返回 false
            return false;
        }

        String isTestCluster = request.getHeaders().getFirst(PerfTestUtils.PERF_TEST_CLUSTER);

        if (StringUtils.isBlank(isTestCluster)) {
            HttpCookie cookie = request.getCookies().getFirst(PerfTestUtils.PERF_TEST_CLUSTER);
            if (cookie != null) {
                isTestCluster = cookie.getValue();
            }
        }

        if (StringUtils.isBlank(isTestCluster)) {
            isTestCluster = request.getQueryParams().getFirst(PerfTestUtils.PERF_TEST_CLUSTER);
        }

        return "1".equals(isTestCluster) || "true".equals(isTestCluster);
    }

}
