package cn.com.duiba.boot.concurrent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import javax.annotation.concurrent.ThreadSafe;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * 支持压测、并发、批量消费的Queue
 *
 * @param <T>
 */
@ThreadSafe
public class BatchConsumeBlockingQueue<T> {

    private static final Logger logger = LoggerFactory.getLogger(BatchConsumeBlockingQueue.class);

    /**
     * 给正常流量使用的queue
     */
    private final LinkedBlockingQueue<T> queue4Normal;

    private final int batchSize;
    private final int maxWaitTimeMillis;
    private final Consumer<List<T>> asyncBatchConsumer;

    /**
     * 最近一次消费时间(正常流量队列)
     */
    private volatile long lastConsumeTimeMillis4Normal = 0;

    private Thread queueThread;
    private volatile boolean isStarted = false;

    /**
     * 构造一个支持压测、并发、批量消费的Queue
     * @param capacity BlockingQueue的最大允许容量, 如果<=0 视为无界队列
     * @param batchSize 一次批量消费的阈值，当已入队的数据个数达到此阈值时，强制调用batchConsumer进行一次消费。
     * @param maxWaitTimeMillis 数据最大延迟消费时间的阈值(ms)，当有数据在队列中存在时间超过此阈值时，强制调用batchConsumer进行一次消费（防止数据存在队列中过久）, 如果此阈值 <= 0则不会触发延时消费（这种情况下数据只会在堆积到达batchSize阈值后才会被消费，这意味着数据可能会在队列中存在非常长的时间）
     * @param asyncBatchConsumer 异步批量消费回调，注意你需要自己确保消息消费成功，如果失败，我们不会再重试消费,另外你需要自行确保asyncBatchConsumer是异步的，一般提交到线程池处理即可。
     */
    public BatchConsumeBlockingQueue(int capacity, int batchSize, int maxWaitTimeMillis, Consumer<List<T>> asyncBatchConsumer) {
        Assert.notNull(asyncBatchConsumer, "batchConsumer must not be null");

        if(capacity <= 0){
            queue4Normal = new LinkedBlockingQueue<>();
        }else {
            queue4Normal = new LinkedBlockingQueue<>(capacity);
        }
        this.batchSize = batchSize;
        this.maxWaitTimeMillis = maxWaitTimeMillis;

        this.asyncBatchConsumer = asyncBatchConsumer;

        if(maxWaitTimeMillis > 0 ) {
            initQueueThread();
        }
    }

    private void initQueueThread(){
        queueThread = new Thread(() -> {
            while (true) {
                try {
                    Thread.sleep(maxWaitTimeMillis);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }

                if (Thread.currentThread().isInterrupted()) {
                    break;
                }

                consumeQueueWhenTimeout();
            }
        });
    }

    private void consumeQueueWhenTimeout(){
        try {
            //批量消费正常流量队列消息
            BlockingQueue<T> queue = determineQueue();
            if (System.currentTimeMillis() - getLastConsumeTimeMillis() >= maxWaitTimeMillis && !queue.isEmpty()) {
                tryBatchConsumeDataWhenTimeout(queue);
            }

            //批量消费压测流量队列消息
            queue = determineQueue();
            if (System.currentTimeMillis() - getLastConsumeTimeMillis() >= maxWaitTimeMillis && !queue.isEmpty()) {
                tryBatchConsumeDataWhenTimeout(queue);
            }
        }catch(Throwable e){
            logger.error("触发消费时发生异常", e);
        }
    }

    /**
     * 启动延时消费线程，每隔maxWaitTimeMillis自动扫描一次有没有入队过久的消息，并触发消费
     */
    public synchronized void start(){
        if(!isStarted) {
            isStarted = true;
            if(maxWaitTimeMillis > 0) {
                queueThread.start();
            }
        }
    }

    /**
     * 停止延时消费线程
     */
    public synchronized void stop(){
        if(isStarted){
            isStarted = false;

            if(maxWaitTimeMillis > 0) {
                queueThread.interrupt();
            }

            //批量消费正常流量队列消息(关闭时影子库的数据就不管了)
            BlockingQueue<T> queue = queue4Normal;
            while (!queue.isEmpty()) {
                tryBatchConsumeDataWhenTimeout(queue);
            }
        }
    }

    /**
     * 根据当前是不是压测决定使用不同的内部queue
     * @return
     */
    private BlockingQueue<T> determineQueue(){
        if(!isStarted){
            throw new IllegalStateException("not started yet!");
        }
        return queue4Normal;
    }

    /**
     * 设置最近消费时间
     */
    private void setLastConsumeTimeMillis(){
        lastConsumeTimeMillis4Normal = System.currentTimeMillis();
    }

    private long getLastConsumeTimeMillis(){
        return lastConsumeTimeMillis4Normal;
    }

    /**
     * 尝试触发批量消费消息（入队消息数量超过batchSize触发）
     * @param queue
     */
    private void tryBatchConsumeDataWhenExceedBatchSize(BlockingQueue<T> queue){
        while(queue.size() >= batchSize){
            List<T> list = new ArrayList<>();
            synchronized (this) {
                if(queue.size() >= batchSize) {
                    queue.drainTo(list, batchSize);//NOSONAR
                }
                setLastConsumeTimeMillis();
            }

            asyncBatchConsumer.accept(list);
        }
    }

    /**
     * 尝试触发批量消费消息(超时触发)
     * @param queue
     */
    private void tryBatchConsumeDataWhenTimeout(BlockingQueue<T> queue){
        if(queue.size() >= 0){
            List<T> list = new ArrayList<>();
            synchronized (this) {
                if(queue.size() >= 0) {
                    queue.drainTo(list, batchSize);//NOSONAR
                }
            }

            asyncBatchConsumer.accept(list);
        }
    }

    /**
     * Inserts the specified element into this queue if it is possible to do
     * so immediately without violating capacity restrictions, returning
     * {@code true} upon success and throwing an
     * {@code IllegalStateException} if no space is currently available.
     * When using a capacity-restricted queue, it is generally preferable to
     * use {@link #offer(Object) offer}.
     *
     * @param t the element to add
     * @return {@code true} (as specified by {@link Collection#add})
     * @throws IllegalStateException if the element cannot be added at this
     *         time due to capacity restrictions
     * @throws ClassCastException if the class of the specified element
     *         prevents it from being added to this queue
     * @throws NullPointerException if the specified element is null
     * @throws IllegalArgumentException if some property of the specified
     *         element prevents it from being added to this queue
     */
    public boolean add(T t) {
        BlockingQueue<T> queue = determineQueue();
        boolean added = queue.add(t);
        if(added){
            tryBatchConsumeDataWhenExceedBatchSize(queue);
        }
        return added;
    }

    /**
     * Inserts the specified element into this queue if it is possible to do
     * so immediately without violating capacity restrictions, returning
     * {@code true} upon success and {@code false} if no space is currently
     * available.  When using a capacity-restricted queue, this method is
     * generally preferable to {@link #add}, which can fail to insert an
     * element only by throwing an exception.
     *
     * @param t the element to add
     * @return {@code true} if the element was added to this queue, else
     *         {@code false}
     * @throws ClassCastException if the class of the specified element
     *         prevents it from being added to this queue
     * @throws NullPointerException if the specified element is null
     * @throws IllegalArgumentException if some property of the specified
     *         element prevents it from being added to this queue
     */
    public boolean offer(T t) {
        BlockingQueue<T> queue = determineQueue();
        boolean added =  queue.offer(t);
        if(added){
            tryBatchConsumeDataWhenExceedBatchSize(queue);
        }
        return added;
    }

    /**
     * Inserts the specified element into this queue, waiting if necessary
     * for space to become available.
     *
     * @param t the element to add
     * @throws InterruptedException if interrupted while waiting
     * @throws ClassCastException if the class of the specified element
     *         prevents it from being added to this queue
     * @throws NullPointerException if the specified element is null
     * @throws IllegalArgumentException if some property of the specified
     *         element prevents it from being added to this queue
     */
    public void put(T t) throws InterruptedException {
        BlockingQueue<T> queue = determineQueue();
        queue.put(t);
        tryBatchConsumeDataWhenExceedBatchSize(queue);
    }

    /**
     * Inserts the specified element into this queue, waiting up to the
     * specified wait time if necessary for space to become available.
     *
     * @param t the element to add
     * @param timeout how long to wait before giving up, in units of
     *        {@code unit}
     * @param unit a {@code TimeUnit} determining how to interpret the
     *        {@code timeout} parameter
     * @return {@code true} if successful, or {@code false} if
     *         the specified waiting time elapses before space is available
     * @throws InterruptedException if interrupted while waiting
     * @throws ClassCastException if the class of the specified element
     *         prevents it from being added to this queue
     * @throws NullPointerException if the specified element is null
     * @throws IllegalArgumentException if some property of the specified
     *         element prevents it from being added to this queue
     */
    public boolean offer(T t, long timeout, TimeUnit unit) throws InterruptedException {
        BlockingQueue<T> queue = determineQueue();
        boolean added = queue.offer(t, timeout, unit);
        if(added){
            tryBatchConsumeDataWhenExceedBatchSize(queue);
        }
        return added;
    }

    /**
     * Returns the number of additional elements that this queue can ideally
     * (in the absence of memory or resource constraints) accept without
     * blocking, or {@code Integer.MAX_VALUE} if there is no intrinsic
     * limit.
     *
     * <p>Note that you <em>cannot</em> always tell if an attempt to insert
     * an element will succeed by inspecting {@code remainingCapacity}
     * because it may be the case that another thread is about to
     * insert or remove an element.
     *
     * @return the remaining capacity
     */
    public int remainingCapacity() {
        return determineQueue().remainingCapacity();
    }

//    @Override
//    public boolean containsAll(Collection<?> c) {
//        return determineQueue().containsAll(c);
//    }

    public boolean addAll(Collection<? extends T> c) {
        if (c == null)
            throw new NullPointerException();
        if (c == this)
            throw new IllegalArgumentException();
        boolean modified = false;
        for (T t : c)
            if (add(t))
                modified = true;
        return modified;
    }

//    @Override
//    public boolean removeAll(Collection<?> c) {
//        return determineQueue().removeAll(c);
//    }

//    @Override
//    public void clear() {
//        determineQueue().clear();
//    }

    /**
     * Returns the number of elements in this collection.  If this collection
     * contains more than <tt>Integer.MAX_VALUE</tt> elements, returns
     * <tt>Integer.MAX_VALUE</tt>.
     *
     * @return the number of elements in this collection
     */
    public int size() {
        return determineQueue().size();
    }

    /**
     * Returns <tt>true</tt> if this collection contains no elements.
     *
     * @return <tt>true</tt> if this collection contains no elements
     */
    public boolean isEmpty() {
        return determineQueue().isEmpty();
    }

    /**
     * Returns {@code true} if this queue contains the specified element.
     * More formally, returns {@code true} if and only if this queue contains
     * at least one element {@code e} such that {@code o.equals(e)}.
     *
     * @param o object to be checked for containment in this queue
     * @return {@code true} if this queue contains the specified element
     * @throws ClassCastException if the class of the specified element
     *         is incompatible with this queue
     *         (<a href="../Collection.html#optional-restrictions">optional</a>)
     * @throws NullPointerException if the specified element is null
     *         (<a href="../Collection.html#optional-restrictions">optional</a>)
     */
    public boolean contains(Object o) {
        return determineQueue().contains(o);
    }

    /**
     * Returns an iterator over the elements in this collection.  There are no
     * guarantees concerning the order in which the elements are returned
     * (unless this collection is an instance of some class that provides a
     * guarantee).
     *
     * @return an <tt>Iterator</tt> over the elements in this collection
     */
    public Iterator<T> iterator() {
        return determineQueue().iterator();
    }

    /**
     * Returns an array containing all of the elements in this collection.
     * If this collection makes any guarantees as to what order its elements
     * are returned by its iterator, this method must return the elements in
     * the same order.
     *
     * <p>The returned array will be "safe" in that no references to it are
     * maintained by this collection.  (In other words, this method must
     * allocate a new array even if this collection is backed by an array).
     * The caller is thus free to modify the returned array.
     *
     * <p>This method acts as bridge between array-based and collection-based
     * APIs.
     *
     * @return an array containing all of the elements in this collection
     */
    public Object[] toArray() {
        return determineQueue().toArray();
    }

    /**
     * Returns an array containing all of the elements in this collection;
     * the runtime type of the returned array is that of the specified array.
     * If the collection fits in the specified array, it is returned therein.
     * Otherwise, a new array is allocated with the runtime type of the
     * specified array and the size of this collection.
     *
     * <p>If this collection fits in the specified array with room to spare
     * (i.e., the array has more elements than this collection), the element
     * in the array immediately following the end of the collection is set to
     * <tt>null</tt>.  (This is useful in determining the length of this
     * collection <i>only</i> if the caller knows that this collection does
     * not contain any <tt>null</tt> elements.)
     *
     * <p>If this collection makes any guarantees as to what order its elements
     * are returned by its iterator, this method must return the elements in
     * the same order.
     *
     * <p>Like the {@link #toArray()} method, this method acts as bridge between
     * array-based and collection-based APIs.  Further, this method allows
     * precise control over the runtime type of the output array, and may,
     * under certain circumstances, be used to save allocation costs.
     *
     * <p>Suppose <tt>x</tt> is a collection known to contain only strings.
     * The following code can be used to dump the collection into a newly
     * allocated array of <tt>String</tt>:
     *
     * <pre>
     *     String[] y = x.toArray(new String[0]);</pre>
     *
     * Note that <tt>toArray(new Object[0])</tt> is identical in function to
     * <tt>toArray()</tt>.
     *
     * @param a the array into which the elements of this collection are to be
     *        stored, if it is big enough; otherwise, a new array of the same
     *        runtime type is allocated for this purpose.
     * @return an array containing all of the elements in this collection
     * @throws ArrayStoreException if the runtime type of the specified array
     *         is not a supertype of the runtime type of every element in
     *         this collection
     * @throws NullPointerException if the specified array is null
     */
    public T[] toArray(T[] a) {
        return determineQueue().toArray(a);
    }

//    @Override
//    public int drainTo(Collection<? super T> c) {
//        return determineQueue().drainTo(c);
//    }
//
//    @Override
//    public int drainTo(Collection<? super T> c, int maxElements) {
//        return determineQueue().drainTo(c, maxElements);
//    }

    @Override
    public String toString() {
        return determineQueue().toString();
    }

}
