package cn.com.duiba.boot.netflix.eureka;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * 该类用于启动初期调用eureka的http接口获取相关数据，此时EurekaClient还未初始化，无法使用
 * Created by guoyanfei .
 * 2022/5/10 .
 */
public final class EurekaHttpUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(EurekaHttpUtils.class);

    /**
     * 最多重试次数
     */
    private static final int NUMBER_OF_RETRIES = 5;

    private static final Splitter SPLITTER = Splitter.on(",");

    private static final CloseableHttpClient HTTP_CLIENT = HttpClientBuilder.create()
                                                                            .setDefaultRequestConfig(RequestConfig.custom().setConnectTimeout(150).setSocketTimeout(150).setConnectionRequestTimeout(100).build())
                                                                            .setMaxConnPerRoute(1)
                                                                            .setMaxConnTotal(10)
                                                                            .evictExpiredConnections()//开启后台线程定时清理失效的连接，每隔10秒主动扫描并逐出超时的连接（超过keepAliveTimeout）
                                                                            .disableAutomaticRetries()//禁止重试
                                                                            .disableCookieManagement()
                                                                            .useSystemProperties()//for proxy
                                                                            .disableRedirectHandling()
                                                                            .setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy(){

                                                                                @Override
                                                                                public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                                                                                    long time = super.getKeepAliveDuration(response, context);
                                                                                    if (time == -1) {
                                                                                        time = 30000;//链接最多空闲30秒
                                                                                    }
                                                                                    return time;
                                                                                }
                                                                            })
                                                                            .build();

    /**
     * 根据eureka的地址 以及 应用名称，获取该应用在eureka中注册的所有UP状态实例的ip:port
     * @param eurekaServerUrl
     * @param serviceId
     * @return
     */
    public static List<String> getUpIpPorts(String eurekaServerUrl, String serviceId) {
        String registeredInfo = getServiceRegisteredInfo(eurekaServerUrl, serviceId);
        JSONArray instaces = parseInstancesArray(registeredInfo);
        if (instaces == null) {
            return Collections.emptyList();
        }

        List<String> ipPorts = new ArrayList<>();
        for (Object instace : instaces) {
            JSONObject inst = (JSONObject) instace;
            // 只捞取UP状态的实例
            if (!"UP".equals(inst.getString("status"))) {
                continue;
            }
            String ip = inst.getString("ipAddr");
            int port = inst.getJSONObject("port").getIntValue("$");
            ipPorts.add(ip + ":" + port);
        }
        return ipPorts;
    }

    /**
     * 根据eureka的地址 以及 应用名称，获取该应用在eureka中注册的所有UP状态的实例的ip和instance
     * @param eurekaServerUrl
     * @param serviceId
     * @return
     */
    public static Map<String, JSONObject> getUpIpInstanceMap(String eurekaServerUrl, String serviceId) {
        String registeredInfo = getServiceRegisteredInfo(eurekaServerUrl, serviceId);
        JSONArray instaces = parseInstancesArray(registeredInfo);
        if (instaces == null) {
            return Collections.emptyMap();
        }

        Map<String, JSONObject> ipMetadataMap = new HashMap<>();

        for (Object instace : instaces) {
            JSONObject inst = (JSONObject) instace;
            // 只捞取UP状态的实例
            if (!"UP".equals(inst.getString("status"))) {
                continue;
            }
            String ip = inst.getString("ipAddr");
            ipMetadataMap.put(ip, inst);
        }
        return ipMetadataMap;
    }

    private static JSONArray parseInstancesArray(String registeredInfo) {
        if (StringUtils.isBlank(registeredInfo)) {
            return null;
        }
        JSONObject infoJSONObj = JSON.parseObject(registeredInfo);
        JSONObject applicationJSONObj = infoJSONObj.getJSONObject("application");
        return applicationJSONObj.getJSONArray("instance");
    }

    /**
     * 根据eureka的地址 以及 应用名称，获取该应用在eureka的注册信息
     * @param eurekaServerUrl
     * @param serviceId
     * @return
     */
    private static String getServiceRegisteredInfo(String eurekaServerUrl, String serviceId) {
        if (StringUtils.isBlank(eurekaServerUrl)) {
            throw new RuntimeException("eureka地址不能为空");
        }
        List<String> eurekaUrls = SPLITTER.splitToList(eurekaServerUrl);
        if (CollectionUtils.isEmpty(eurekaUrls)) {
            throw new RuntimeException("eureka地址错误, eurekaServerUrl=" + eurekaServerUrl);
        }

        for (int retry = 0; retry < NUMBER_OF_RETRIES; retry++) {
            String currentEurekaUrl = eurekaUrls.get(retry % eurekaUrls.size());
            HttpUriRequest getRequest = new HttpGet(currentEurekaUrl + "apps/" + serviceId);
            getRequest.addHeader("Accept", "application/json");
            getRequest.addHeader("Accept-Encoding", "gzip");
            try (CloseableHttpResponse response = HTTP_CLIENT.execute(getRequest)) {
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode != 200) {
                    throw new Exception("status code = " + statusCode);
                }
                return EntityUtils.toString(response.getEntity());
            } catch (Exception e) {
                LOGGER.warn("通过http方式调用eureka接口获取注册信息异常, 应用名称: {}, error msg: {}", serviceId, e.getMessage());
            }

        }
        LOGGER.error("通过http方式调用eureka接口获取注册信息异常, 重试" + NUMBER_OF_RETRIES + "次后依然失败");
        return null;
    }

}
