package cn.com.duiba.boot.ext.autoconfigure.rocketmq;

import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.*;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;

/**
 * RocketMQ生产者/消费者自动配置
 * Created by wenqi.huang on 2017/04/18.
 */
@Configuration
@EnableConfigurationProperties(RocketMqProperties.class)
public class RocketMqAutoConfiguration {

    protected static abstract class RocketMqClientConfiguration{

        @Autowired
        protected RocketMqProperties rocketMqProperties;

    }

    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class})
    @ConditionalOnMissingBean(name={"rocketMqProducer"})
    @ConditionalOnProperty(name="duiba.rocketmq.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class RocketMqProducerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqProducer", destroyMethod = "shutdown")
        public DefaultMQProducer rocketMqProducer() throws MQClientException {
            /**
             * 一个应用创建一个Producer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             */
            DefaultMQProducer p = new DefaultMQProducer();

            /**
             * producerGroup 这个概念发送普通的消息时，作用不大，但是发送分布式事务消息时，比较关键，
             * 因为服务器会回查这个Group下的任意一个Producer
             * 建议把producerGroup设置成应用名
             */
            p.setProducerGroup(rocketMqProperties.getProducer().getGroup());
            p.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            //p.setInstanceName("Producer");
            p.setSendMsgTimeout(rocketMqProperties.getProducer().getSendMsgTimeoutMillis());//单位：ms
            /**
             * Producer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             * 注意：切记不可以在每次发送消息时，都调用start方法
             */
            p.start();

            return p;
        }
    }

    @Configuration
    @ConditionalOnClass({DefaultMQPushConsumer.class})
    @ConditionalOnMissingBean(name={"rocketMqConsumer"})
    @ConditionalOnProperty(name="duiba.rocketmq.consumer.enable", havingValue = "true", matchIfMissing = false)
    protected static class RocketMqConsumerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqConsumer")
        public FactoryBean<DefaultMQPushConsumer> rocketMqConsumer() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new RocketMqConsumerFactoryBean(rocketMqProperties);
        }
    }

    private static class RocketMqConsumerFactoryBean implements FactoryBean<DefaultMQPushConsumer>,InitializingBean,DisposableBean{

        @Resource(name="bootRocketMqMessageListener",type = MessageListener.class)
        private MessageListener rocketMqMessageListener;

        private DefaultMQPushConsumer rocketMqConsumer;
        private RocketMqProperties rocketMqProperties;

        public RocketMqConsumerFactoryBean(RocketMqProperties rocketMqProperties){
            this.rocketMqProperties = rocketMqProperties;
        }

        @Override
        public void afterPropertiesSet() throws Exception {
            /**
             * 一个应用创建一个Consumer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             * 注意：consumerGroup 需要由应用来保证唯一
             * consumerGroup建议使用应用名。RocketMQ使用consumerGroup作为负载均衡的粒度，
             * 所以同一个consumerGroup订阅的Topic必须完全相同。
             * 考虑这样的场景，活动中心之前已经订阅了一些topic，某次发布新版需要增加一个topic的订阅，发布一部分服务器的情况下，新的topic的某些queue不会被消费，所以需要确保活动中心全发。某个版本如果新增了topic，则不能做灰度发布。考虑到代码可读性和性能，这个代价是值得的
             */
            //DefaultMQPullConsumer consumer = new DefaultMQPullConsumer();
            DefaultMQPushConsumer consumer = new DefaultMQPushConsumer(rocketMqProperties.getConsumer().getGroup());
            //consumer.setConsumerGroup();
            consumer.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            consumer.setMaxReconsumeTimes(rocketMqProperties.getConsumer().getMaxReconsumeTimes());//最大重复消费次数，超过这个次数还没被确认则丢弃消息，请根据消息重要程度设置
            if(rocketMqProperties.getConsumer().getConsumeThreadNums() > 0) {
                consumer.setConsumeThreadMin(rocketMqProperties.getConsumer().getConsumeThreadNums());//消费最小线程数
                consumer.setConsumeThreadMax(Math.max(rocketMqProperties.getConsumer().getConsumeThreadNums(), 64));//消费最大线程数
            }
            //consumer.setConsumeTimeout(15);
            //consumer.setPullBatchSize(32);
            consumer.setConsumeMessageBatchMaxSize(1);//监听器每次接受本地队列的消息是多少条(从服务器获取还是多条的，只是本地MessageListener每次处理只有一条)
            //consumer.setMessageModel(MessageModel.BROADCASTING);
            //consumer.setPullInterval();
            //consumer.setInstanceName("Consumber");

            String topics = rocketMqProperties.getConsumer().getTopics();
            String[] topicArr = topics.split(",");

            for(String topic : topicArr) {
                consumer.subscribe(topic, "*");
            }
            //尽量使用MessageListenerConcurrently来消费，吞吐量更高，消费无顺序
            //当对消息消费顺序有严格要求时，请使用MessageListenerOrderly
            if(rocketMqMessageListener instanceof MessageListenerConcurrently) {
                consumer.registerMessageListener((MessageListenerConcurrently)rocketMqMessageListener);
            }else if(rocketMqMessageListener instanceof MessageListenerOrderly) {
                consumer.registerMessageListener((MessageListenerOrderly)rocketMqMessageListener);
            }

            /**
             * Consumer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             */
            consumer.start();

            rocketMqConsumer = consumer;
        }

        @Override
        public DefaultMQPushConsumer getObject() throws Exception {
            return rocketMqConsumer;
        }

        @Override
        public Class<?> getObjectType() {
            return DefaultMQPushConsumer.class;
        }

        @Override
        public boolean isSingleton() {
            return true;
        }

        @Override
        public void destroy() throws Exception {
            rocketMqConsumer.shutdown();
        }
    }
}
