package cn.com.duibaboot.ext.autoconfigure.web.mvc;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.stereotype.Controller;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.condition.RequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 自定义的CustomRequestMappingHandlerMapping, 主要作用是方便重用FeignClient。并对AdvancedFeignClient做特殊处理
 * 目前官方spring mvc支持在父类方法中定义RequestMapping，子类可以不定义RequestMapping，然而如果此时父类参数加了RequestParam等注解，是不会生效的，必须写在子类中。这样的话父类和子类都必须写，非常繁琐。
 * 这个CustomRequestMappingHandlerMapping的作用就是让spring到声明RequestMapping的方法上去找参数注解<br/>
 * 对AdvancedFeignClient做的特殊处理：自动侦查方法，使用方法名作为path（对于类，也会默认使用类名作为path），除非用户自己使用RequestMapping指定。
 *
 * Created by wenqi.huang on 2017/7/17.
 */
public class CustomRequestMappingHandlerMapping extends RequestMappingHandlerMapping {

    @Override
    protected boolean isHandler(Class<?> beanType) {
        if(AnnotatedElementUtils.hasAnnotation(beanType, Controller.class)){//hasAnnotation也会寻找父类/接口中的注解
            return true;
        }

        if(AnnotatedElementUtils.hasAnnotation(beanType, RequestMapping.class) && !beanType.isInterface()){
            //防止把FeignClient的fallback类也识别成了mvchandler，
            if(AnnotatedElementUtils.hasAnnotation(beanType, FeignClient.class) || AnnotatedElementUtils.hasAnnotation(beanType, AdvancedFeignClient.class)){
                return false;
            }else{
                return true;
            }
        }else{
            return false;
        }
    }

    /**
     * 这个方法用于识别一个方法是不是可以转换为RequestMapping，可以的话则转换。
     * @param method
     * @param handlerType
     * @return
     */
    @Override
    protected RequestMappingInfo getMappingForMethod(Method method, Class<?> handlerType) {
        boolean isAdvancedFeignClientMode = false;
        Class[] interfaces = handlerType.getInterfaces();
        int interfaceCntWithFCAnno = 0;
        for(Class clazz : interfaces){
            if(clazz.isAnnotationPresent(AdvancedFeignClient.class)){
                isAdvancedFeignClientMode = true;
                interfaceCntWithFCAnno ++;
            }
        }
        //一个类只能有一个注解了@AdvancedFeignClient的接口， 如果多于一个，则报错.
        if(interfaceCntWithFCAnno > 1){
            throw new IllegalStateException("the class:"+handlerType.getName()+" can only have one interface annotated with @AdvancedFeignClient");
        }
        if(!isAdvancedFeignClientMode){//没有接口注解了AdvancedFeignClient，使用springmvc默认行为来处理@RequestMapping
            return super.getMappingForMethod(method, handlerType);
        }
        //至少有一个接口注解了AdvancedFeignClient，则进入自动侦查模式，自动把注解了AdvancedFeignClient所在接口上的方法识别为RequestMapping，方法名作为path，所以不能有两个方法名字一样（或者可以手动注解@RequestMapping指定path）
        Method methodInInterface = null;
        for(Class clazz : interfaces){
            if(clazz.isAnnotationPresent(AdvancedFeignClient.class)){
                try {
                    methodInInterface = clazz.getMethod(method.getName(), method.getParameterTypes());
                    break;
                } catch (NoSuchMethodException e) {
                    //Ignore
                }
            }
        }
        if(methodInInterface == null || methodInInterface.isDefault()){//default方法在客户端用接口调用的时候不会调用rpc，故不能扫描为mvc方法
            return null;
        }

        //先查找改方法上是否有RequestMapping注解，如果有的话，优先使用方法上的
        RequestMappingInfo info = createRequestMappingInfo(methodInInterface);

        //如果没有，则自动识别,使用方法名作为path，method默认为任意值。
        if(info == null){
            //根据方法信息构造一个RequestMapping实例
            String path = method.getName();
            RequestMapping requestMapping = genRequestMapping(path);
            info = createRequestMappingInfo(requestMapping, method);
        }else if(info.getPatternsCondition().getPatterns().isEmpty() || StringUtils.isEmpty(info.getPatternsCondition().getPatterns().iterator().next())){
            //如果RequestMapping的path为空，则以方法名作为path
            String path = method.getName();
            RequestMapping requestMapping = genRequestMapping(path);
            RequestMappingInfo otherInfo = createRequestMappingInfo(requestMapping, method);
            info = otherInfo.combine(info);
        }

        //到对应的接口上寻找RequestMapping注解并合并
        RequestMappingInfo typeInfo = createRequestMappingInfo(methodInInterface.getDeclaringClass());
        if(typeInfo == null){
            //如果接口上没有RequestMapping注解，则使用类名作为path(类名首字母小写)
            String path = StringUtils.uncapitalize(methodInInterface.getDeclaringClass().getSimpleName());
            RequestMapping requestMapping = genRequestMapping(path);
            typeInfo = createRequestMappingInfo(requestMapping, methodInInterface.getDeclaringClass());
        }else if(typeInfo.getPatternsCondition().getPatterns().isEmpty() || StringUtils.isEmpty(typeInfo.getPatternsCondition().getPatterns().iterator().next())){
            //如果RequestMapping的path为空，则以类名作为path(类名首字母小写)
            String path = StringUtils.uncapitalize(methodInInterface.getDeclaringClass().getSimpleName());
            RequestMapping requestMapping = genRequestMapping(path);
            RequestMappingInfo otherTypeInfo = createRequestMappingInfo(requestMapping, methodInInterface.getDeclaringClass());
            typeInfo = otherTypeInfo.combine(typeInfo);
        }

        info = typeInfo.combine(info);

        return info;
    }

    private RequestMapping genRequestMapping(String path){
        return new RequestMapping(){

            @Override
            public String[] value() {
                return new String[]{path};
            }

            @Override
            public String[] path() {
                return value();
            }

            @Override
            public RequestMethod[] method() {
                return new RequestMethod[0];
//                return new RequestMethod[]{RequestMethod.POST};
            }

            @Override
            public String[] params() {
                return new String[0];
            }

            @Override
            public String[] headers() {
                return new String[0];
            }

            @Override
            public String[] consumes() {
                return new String[0];
            }

            @Override
            public String[] produces() {
                return new String[0];
            }

            @Override
            public String name() {
                return "";
            }

            @Override
            public Class<? extends Annotation> annotationType() {
                return RequestMapping.class;
            }
        };
    }

    private RequestMappingInfo createRequestMappingInfo(RequestMapping requestMapping, AnnotatedElement element) {
        RequestCondition<?> condition = (element instanceof Class ?
                getCustomTypeCondition((Class<?>) element) : getCustomMethodCondition((Method) element));
        return (requestMapping != null ? createRequestMappingInfo(requestMapping, condition) : null);
    }

    //copy from super.createRequestMappingInfo
    private RequestMappingInfo createRequestMappingInfo(AnnotatedElement element) {
        RequestMapping requestMapping = AnnotatedElementUtils.findMergedAnnotation(element, RequestMapping.class);
        RequestCondition<?> condition = (element instanceof Class ?
                getCustomTypeCondition((Class<?>) element) : getCustomMethodCondition((Method) element));
        return (requestMapping != null ? createRequestMappingInfo(requestMapping, condition) : null);
    }

    @Override
    protected HandlerMethod getHandlerInternal(HttpServletRequest request) throws Exception {
        HandlerMethod m = super.getHandlerInternal(request);
        if(m == null){
            return m;
        }
        MethodParameter[] parameters = m.getMethodParameters();
        if(parameters == null){
            return m;
        }

        injectParameterAnnotations(parameters);

        return m;
//        return new CustomHandlerMethod(m);
    }

    //TODO 已经测试过与spring-boot 1.4.5/1.4.7版本(对应spring4.3.9）的兼容性，如果换用其他版本，这里可能会有问题，需要跟进
    /**
     * 这个方法的主要作用：判断如果RequestMapping没有注解在子类方法上，则到父类或父接口中寻找注解了RequestMapping的方法，并使用这个方法参数中的RequestParam/RequestBody等注解
     * @param parameters
     * @throws IllegalAccessException
     * @throws InvocationTargetException
     */
    private void injectParameterAnnotations(MethodParameter[] parameters) throws IllegalAccessException, InvocationTargetException {

        //这里获得的MethodParameter实际类型是HandlerMethod.HandlerMethodParameter
        for(MethodParameter p : parameters){
            Method method = p.getMethod();
            if(method == null){
                continue;
            }
            Field field = ReflectionUtils.findField(p.getClass(), "parameterAnnotations");//p.getParameterAnnotations()
            field.setAccessible(true);
            //第一次调用肯定是null，调用过一次之后设置进去就不会为null了
            if(field.get(p) == null){
                Method requestMappingAnnotatedMethod = null;
                if(!isRequestMappingAnnotated(method)) {//优先到当前类或者父类中注解了@RequestMapping的方法中找对应参数的注解
                    Class[] interfaces = method.getDeclaringClass().getInterfaces();
                    Class superClass = method.getDeclaringClass().getSuperclass();
                    List<Class> classesToSearch = new ArrayList<>();
                    if(superClass != null || superClass != Object.class) {
                        classesToSearch.add(superClass);
                    }
                    classesToSearch.addAll(Arrays.asList(interfaces));
                    for (Class clazz : classesToSearch) {
                        try {
                            Method superMethod = clazz.getMethod(method.getName(), method.getParameterTypes());
                            if (isRequestMappingAnnotated(superMethod)){
                                requestMappingAnnotatedMethod = superMethod;
                                break;
                            }
                        } catch (NoSuchMethodException e) {
                            //Ignore
                        }
                    }
                }
                if(requestMappingAnnotatedMethod != null){
                    //优先从父类加载@RequestParam等注解的数据
                    Annotation[] parameterAnnotations = null;
                    Annotation[][] annotationArray = requestMappingAnnotatedMethod.getParameterAnnotations();
                    if (p.getParameterIndex() >= 0 && p.getParameterIndex() < annotationArray.length) {
                        //HandlerMethod.HandlerMethodParameter类中调用了adaptAnnotationArray方法，所以这里必须调用
                        //因为adaptAnnotationArray方法是protected，所以只能反射调用
//                        parameterAnnotations = p.adaptAnnotationArray(annotationArray[p.getParameterIndex()]);

                        Method adaptAnnotationArrayMethod = ReflectionUtils.findMethod(p.getClass(), "adaptAnnotationArray", Annotation[].class);
                        adaptAnnotationArrayMethod.setAccessible(true);
                        Object[] args = new Object[1];
                        Annotation[] anno = annotationArray[p.getParameterIndex()];
                        args[0] = anno;
                        parameterAnnotations = (Annotation[])adaptAnnotationArrayMethod.invoke(p, args);
                    }
                    else {
                        parameterAnnotations = new Annotation[0];
                    }
                    field.set(p, parameterAnnotations);

                    //确保这个方法存在
                    p.getParameterAnnotations();
                }
            }

        }
    }

    /**
     * 判断方法上是否有RequestMapping(GetMapping/PostMapping)注解
     */
    private boolean isRequestMappingAnnotated(Method m){
        Annotation[] arr = m.getAnnotations();
        for(Annotation a : arr){
            if(a instanceof RequestMapping || a.getClass().getAnnotation(RequestMapping.class)!=null){//后面这个表示GetMapping/PostMapping等注解
                return true;
            }
        }
        return false;
    }
}
