package cn.com.duibaboot.ext.autoconfigure.web.mvc;

import cn.com.duiba.boot.exception.BizException;
import com.alibaba.fastjson.JSON;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.web.ErrorAttributes;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.UndeclaredThrowableException;
import java.util.Map;

/**
 * 这个HandlerExceptionResolver(全局mvc异常处理)用于对BizException类进行特殊处理，返回的错误json字符串中多加入一个code字段
 * <br/>
 * 并且这个异常不会打印到日志中。
 */
@Order(Ordered.LOWEST_PRECEDENCE - 10)
public class BizExceptionResolver implements HandlerExceptionResolver, Ordered {

    @Autowired
    private ErrorAttributes errorAttributes;

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE - 10;
    }

    @Override
    public ModelAndView resolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        //只对BizException进行特殊处理, 比BasicErrorController的处理多返回一个code参数
        //普通的web请求会把BizException包装为UndeclaredThrowableException，Feign的请求不会
//        if(ex instanceof UndeclaredThrowableException){
//            Throwable t = ((UndeclaredThrowableException)ex).getUndeclaredThrowable();
//            if(t instanceof Exception){
//                ex = (Exception)t;
//            }
//        }

        if(ex instanceof BizException){
            Map<String, Object> body = getErrorAttributes(request, false);
            body.put("status", HttpStatus.INTERNAL_SERVER_ERROR.value());
            body.put("error", HttpStatus.INTERNAL_SERVER_ERROR.getReasonPhrase());
            if(((BizException) ex).getCode() != null) {
                body.put("code", ((BizException) ex).getCode());
            }
            body.put("exception", ex.getClass().getName());
            body.put("message", ex.getMessage());
            String jsonStr = JSON.toJSONString(body);
            try {
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                IOUtils.write(jsonStr, response.getOutputStream(), Charsets.UTF_8);
                IOUtils.closeQuietly(response.getOutputStream());
            } catch (IOException e) {
                throw new RuntimeException(e.getMessage(), e);
            }
            return new ModelAndView();
        }

        //返回null表示不处理.
        return null;
    }

    private Map<String, Object> getErrorAttributes(HttpServletRequest request,
                                                     boolean includeStackTrace) {
        RequestAttributes requestAttributes = new ServletRequestAttributes(request);
        return this.errorAttributes.getErrorAttributes(requestAttributes,
                includeStackTrace);
    }

}
