package cn.com.duibaboot.ext.autoconfigure.accesslog;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Charsets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.AntPathMatcher;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by xuhengfei on 16/12/27.
 */
public class AccessLogFilter implements Filter{

    public static ThreadLocal<Long> AppId=new ThreadLocal<>();
    public static ThreadLocal<Long> ConsumerId=new ThreadLocal<>();
    public static ThreadLocal<Map<String,String>> LogCookie=new ThreadLocal<>();
    public static ThreadLocal<Map<String,Object>> EX=new ThreadLocal<>();
    public static ThreadLocal<Map<String,String>> OverWrite=new ThreadLocal<>();

    public static final String OW_HOST="host";

    public static final String ModeBlack="black";//黑名单模式
    public static final String ModeWhite="white";//白名单模式


    private String mode=ModeBlack;
    private Set<String> inhosts=new HashSet<>();

    private Set<String> blackPaths=new HashSet<>();
    private Set<String> blackUriPatterns = new HashSet<>();
    private AntPathMatcher pathMatcher = new AntPathMatcher();

    /**
     * 加载配置项
     * @param properties
     */
    protected void loadConfig(Properties properties){
        mode=properties.getProperty("accesslog.mode");
        String hoststring=properties.getProperty("accesslog.hosts");
        if(hoststring!=null){
            List<String> hosts=Arrays.asList(hoststring.trim().split(","));
            if(hosts!=null && !hosts.isEmpty()){
                inhosts=new HashSet<>(hosts);
            }
        }
        String blackpaths=properties.getProperty("accesslog.blackpaths");
        if(blackpaths!=null){
            List<String> paths=Arrays.asList(blackpaths.trim().split(","));
            if(paths!=null && !paths.isEmpty()){
                for(String path : paths){
                    if(path != null && !path.contains("*")){
                        blackPaths.add(path);
                    }else if(path != null && path.contains("*")){
                        blackUriPatterns.add(path);
                    }
                }
            }
        }
    }

    private String getHost(HttpServletRequest req){
        if(OverWrite.get()!=null){
            String host=OverWrite.get().get(OW_HOST);
            if(host!=null){
                return host;
            }
        }

        return req.getHeader("host");
    }

    /**
     * 是否需要进行日志打印
     * @return
     */
    private boolean needLog(HttpServletRequest req){
        String host=getHost(req);
        if(ModeBlack.equals(mode)){
            if(inhosts.contains(host)){
                return false;
            }
        }else if(ModeWhite.equals(mode)){
            if(!inhosts.contains(host)){
                return false;
            }
        }

        String path=clearRequestURI(req.getRequestURI());
        if(blackPaths.contains(path)){
            return false;
        }
        for(String pattern : blackUriPatterns){
            if(this.pathMatcher.match(pattern, path)){
                return false;
            }
        }

        return true;

    }

    /**
     * 添加cookie信息到线程变量,日志优先从线程变量中获取
     * @param key
     * @param value
     */
    public static void putLogCookie(String key,String value){
        Map<String,String> map=LogCookie.get();
        if(map==null){
            map=new HashMap<>();
            LogCookie.set(map);
        }
        map.put(key,value);
    }

    /**
     * 添加额外信息到线程变量,日志从线程变量获取信息进行打压输出
     * @param key
     * @param value
     */
    public static void putExPair(String key,Object value){
        Map<String,Object> ex=EX.get();
        if(ex==null){
            ex=new HashMap<>();
            EX.set(ex);
        }
        ex.put(key,value);
    }
    /**
     * 添加覆盖信息到线程变量,打印日志时优先从此处获取
     * @param key
     * @param value
     */
    public static void putOverWritePair(String key,String value){
        Map<String,String> map=OverWrite.get();
        if(map==null){
            map=new HashMap<>();
            OverWrite.set(map);
        }
        map.put(key,value);
    }

    private static Logger AccessLog= LoggerFactory.getLogger("duiba_access_log");

    private static Logger log=LoggerFactory.getLogger(AccessLogFilter.class);

    private static final String DUIBA_PERF="_duibaPerf";


    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {

        Long startTime= System.currentTimeMillis();
        Map<String,String> collCookie=new HashMap<>();
        try {
            try {
                //在执行filter之前种cookie,避免resp被提前关闭,无法写入cookie
                addCookieIfNeed((HttpServletRequest) servletRequest, (HttpServletResponse) servletResponse, collCookie);
            }catch(Exception e){
                log.error("addCookieIfNeed error");
            }
            filterChain.doFilter(servletRequest, servletResponse);

        }finally {
            try {
                HttpServletRequest req = (HttpServletRequest) servletRequest;
                //判断是否性能压测url
                boolean isPerfUrl=false;

                String _duibaPerf=req.getParameter(DUIBA_PERF);
                if(_duibaPerf!=null && "1".equals(_duibaPerf)){
                    isPerfUrl=true;
                }

                if(!isPerfUrl){
                    Cookie[] cookies = req.getCookies();
                    if(cookies!=null){
                        for(Cookie c:cookies){
                            if(DUIBA_PERF.equals(c.getName()) && "1".equals(c.getValue())){
                                isPerfUrl=true;
                                break;
                            }
                        }
                    }

                }


                if(!isPerfUrl && needLog(req)){
                    processAccessLog(req,(HttpServletResponse)servletResponse,startTime,collCookie);
                }


            }catch (Exception e){
                log.error("AccessLogFilter process error, message="+e.getMessage());
            }

            AppId.remove();
            ConsumerId.remove();
            LogCookie.remove();
            EX.remove();
            OverWrite.remove();
        }
    }

    private void addCookieIfNeed(HttpServletRequest req,HttpServletResponse resp,Map<String,String> collCookie){
        Map<String,String[]> params=req.getParameterMap();
        String host=req.getHeader("host");
        String cookieHost=getCookieHost(host);

        for(String key:params.keySet()){
            if(key.startsWith("tck_")){
                String[] v=key.split("_");
                if(v.length==3){
                    String verify=null;
                    try {
                        verify = MD5.md5(v[1]);
                    }catch (Exception e){
                        return;
                    }
                    if(verify.endsWith(v[2])){
                        String name=v[1];
                        String value=req.getParameter(key);
                        //验证通过
                        String cookiekey="_coll_"+name;
                        collCookie.put(cookiekey,value);
                        //写入cookie

                        Cookie cookie = new Cookie(cookiekey, value);
                        if(cookieHost!=null) {
                            cookie.setDomain(cookieHost);
                        }
                        cookie.setPath("/");
                        try {
                            resp.addCookie(cookie);
                        }catch (Exception e){
                            log.error("addCookie error,cookieKey="+cookiekey+",value="+value,e);
                        }
                    }
                }
            }
        }
    }


    private void processAccessLog(HttpServletRequest req, HttpServletResponse resp,long startTime,Map<String,String> collCookie) throws Exception{
        //只记录get post请求
        if(!"get".equalsIgnoreCase(req.getMethod()) && !"post".equalsIgnoreCase(req.getMethod())){
            return;
        }

        Map<String, Object> map = new HashMap<>();
        map.put("url_host", getHost(req));
        map.put("url_path", clearRequestURI(req.getRequestURI()));
        map.put("url_query", req.getQueryString());

        map.put("http_method",req.getMethod());
        map.put("rc",resp.getStatus());
        map.put("rt",(System.currentTimeMillis()-startTime));
        map.put("mip", NetUtils.getLocalIp());//记录服务器IP
        if(EX.get()!=null){
            map.put("ex",EX.get());
        }

        //如果url中有callback参数,认为此url是一个JSONP请求,标记为POST类型,不作为PV统计
        String callback=req.getParameter("callback");
        if(callback!=null && !"".equals(callback.trim())){
            map.put("http_method","POST");
        }



        Cookie[] cookies = req.getCookies();
        Map<String, Cookie> cookieMap = new HashMap<>();
        if(cookies!=null){
            for (Cookie c : cookies) {
                cookieMap.put(c.getName(), c);
            }
        }


        //优先从线程变量中取数据
        Long consumerId=ConsumerId.get();
        if(consumerId!=null){
            map.put("consumer_id",consumerId);
        }

        Long appId=AppId.get();
        if(appId!=null){
            map.put("app_id",appId);
        }

        Cookie _ac = cookieMap.get("_ac");
        if (_ac != null){
            if(consumerId==null || appId==null){//如果还没有取到,尝试从cookie取数据
                try {
                    String json = new String(SecurityUtils.decodeBase64(_ac.getValue()), Charsets.UTF_8);
                    if(json!=null){
                        JSONObject object = JSONObject.parseObject(json);
                        if(appId==null){
                            map.put("app_id",object.getLong("aid"));
                        }
                        if(consumerId==null){
                            map.put("consumer_id",object.getLong("cid"));
                        }
                    }
                }catch(Exception e){
                    log.warn("_ac decode fail , _ac= "+_ac.getValue());
                }
            }

        }


        for(String key:cookieMap.keySet()){
            if(key.startsWith("_coll_")){
                collCookie.put(key,cookieMap.get(key).getValue());
            }
        }

        Map<String,String[]> params=req.getParameterMap();
        String host=req.getHeader("host");
        String cookieHost=getCookieHost(host);

        for(String key:params.keySet()){
            if(key.startsWith("tck_")){
                String[] v=key.split("_");
                if(v.length==3){
                    String verify=MD5.md5(v[1]);
                    if(verify.endsWith(v[2])){
                        String name=v[1];
                        String value=req.getParameter(key);
                        //验证通过
                        String cookiekey="_coll_"+name;
                        collCookie.put(cookiekey,value);
                        //写入cookie

                        Cookie cookie = new Cookie(cookiekey, value);
                        if(cookieHost != null) {
                            cookie.setDomain(cookieHost);
                        }
                        cookie.setPath("/");
                        resp.addCookie(cookie);

                    }
                }
            }
        }

        //特殊处理
        String slotId=req.getParameter("adslotId");
        if(slotId!=null){
            collCookie.put("_coll_slot",slotId);
        }
        String deviceId=req.getParameter("deviceId");
        if(deviceId!=null){
            collCookie.put("_coll_device",deviceId);
        }

        //支持线程变量中取cookie进行日志打印,便于不同系统的一些定制需求,此为cookie的最高优先级获取方法
        Map<String,String> logCookie=LogCookie.get();
        if(logCookie!=null){
            for(String key:logCookie.keySet()){
                collCookie.put(key,logCookie.get(key));
            }
        }

        if(!collCookie.isEmpty()){
            map.put("cookie",collCookie);
        }


        String ua = req.getHeader("user-agent");
        if (ua != null && ua.length() > 500) {
            ua = ua.substring(0, 499);
        }
        if (ua != null) {
            map.put("user_agent", ua);
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        map.put("time", sdf.format(new Date()));
        String referer = req.getHeader("referer");
        if (referer != null) {
            map.put("referer", referer);
        }
        String ip = RequestTool.getIpAddr(req);
        if (ip != null) {
            map.put("ip", ip);
        }


        AccessLog.info(JSON.toJSONString(map));
    }

    /**
     * requestURI中可能出现连续两个/，比如//activity//index，需要处理下转为一个/
     * @param requestURI
     * @return
     */
    private String clearRequestURI(String requestURI){
        if(StringUtils.isBlank(requestURI)){
            return requestURI;
        }

        return StringUtils.replace(requestURI, "//", "/");
    }

    private Map<String,String> cookieHosts=new ConcurrentHashMap<String,String>();

    /**
     * 根据host查询 通配符cookie地址
     * @param host
     * @return
     */
    private String getCookieHost(String host){
        if(host==null){
            return null;
        }
        if(host.contains(":")){
            host=host.substring(0,host.indexOf(":"));
        }
        String cookieHost=cookieHosts.get(host);
        if(cookieHost!=null){
            return cookieHost;
        }
        String hostLow=host.toLowerCase();
        if(hostLow.endsWith(".duiba.com.cn") ){
            cookieHost="duiba.com.cn";
        }else if(hostLow.endsWith(".dui88.com")){
            cookieHost="dui88.com";
        }else if(hostLow.endsWith(".duibar.com")){
            cookieHost="duibar.com";
        }else if(hostLow.endsWith(".tuia.cn")) {
            cookieHost = "tuia.cn";
        }

        if(cookieHost!=null){
            cookieHosts.put(host,cookieHost);
        }
        return cookieHost;
    }

    @Override
    public void destroy() {

    }

}
