package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import cn.com.duiba.boot.utils.JarVersionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.cloud.netflix.eureka.EurekaClientAutoConfiguration;
import org.springframework.util.ClassUtils;

/**
 * eurekaClient实例在 @ConditionalOnRefreshScope 情况下每次 refresh，会销毁重新创建。实例会取消注册，再重新注册到eureka。导致当前应用refresh时，其他应用从eureka获取该应用实例的时候存在找不到的情况
 * 此处扩展后，refresh不再会重新创建eurekaClient实例
 * Created by gyf .
 * 2018/4/27 .
 */
public class EurekaClientBeanDefinitionRegistryPostProcessor implements BeanDefinitionRegistryPostProcessor {

    private static final Logger logger = LoggerFactory.getLogger(EurekaClientBeanDefinitionRegistryPostProcessor.class);

    private static boolean isVersionMatch;

    static{
        //spring-cloud 低版本的情况，如果把eurekaClient改为singleton，会导致被/refresh时不会再重新注册到eurekaServer上，从而导致问题
        boolean isClassExists = ClassUtils.isPresent("org.springframework.cloud.netflix.eureka.EurekaClientAutoConfiguration", EurekaClientBeanDefinitionRegistryPostProcessor.class.getClassLoader());
        if(isClassExists) {
            //1.4.3.RELEASE 对应spring-cloud的 Edgware.SR2 版本
            //1.4.4.RELEASE 对应spring-cloud的 Edgware.SR3 版本
            String version = JarVersionUtils.getJarVersion(EurekaClientAutoConfiguration.class);
            boolean isVersionMatch = version.equals("1.4.3.RELEASE") || version.equals("1.4.4.RELEASE");
            EurekaClientBeanDefinitionRegistryPostProcessor.isVersionMatch = isVersionMatch;
            if (!isVersionMatch) {
                logger.error("请升级spring-cloud到如下版本： mavenBom 'org.springframework.cloud:spring-cloud-dependencies:Edgware.SR2', (如当前使用的版本已经高于此版本，则通知架构组修改对此版本的兼容性)");
            }
        }
    }

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        if(isVersionMatch) {
            BeanDefinition eurekaClientBeanDefinition = null;
            try {
                eurekaClientBeanDefinition = registry.getBeanDefinition("eurekaClient");
            } catch (NoSuchBeanDefinitionException e) {
                //Ignore
                return;
            }

            if (eurekaClientBeanDefinition == null) {
                return;
            }
            eurekaClientBeanDefinition.setScope(BeanDefinition.SCOPE_SINGLETON);

            BeanDefinition eurekaClientOriginatingBeanDefinition = eurekaClientBeanDefinition.getOriginatingBeanDefinition();
            if (eurekaClientOriginatingBeanDefinition == null) {
                return;
            }
            eurekaClientOriginatingBeanDefinition.setScope(BeanDefinition.SCOPE_SINGLETON);
        }
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {

    }

}
