/*
 * Copyright 2013-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.etcd.cloud;

import cn.com.duibaboot.ext.autoconfigure.initserver.MainApplicationContextHolder;
import mousio.client.ConnectionState;
import mousio.client.retry.RetryPolicy;
import mousio.etcd4j.EtcdClient;
import mousio.etcd4j.responses.EtcdException;
import mousio.etcd4j.responses.EtcdHealthResponse;
import mousio.etcd4j.responses.EtcdKeysResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.context.refresh.ContextRefresher;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.EnumerablePropertySource;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author Luca Burgazzoli
 * @author Spencer Gibb
 */
public class EtcdPropertySource extends EnumerablePropertySource<EtcdClient> {

	private static final Logger logger = LoggerFactory.getLogger(EtcdPropertySource.class);

	private final Map<String, String> properties;
	private final String prefix;
	private final EtcdConfigProperties config;

	public EtcdPropertySource(String root, EtcdClient source, EtcdConfigProperties config) {
		super(root, source);
		this.properties = new HashMap<>();
		this.prefix = root.startsWith(EtcdConstants.PATH_SEPARATOR) ? root
				+ EtcdConstants.PATH_SEPARATOR : EtcdConstants.PATH_SEPARATOR + root
				+ EtcdConstants.PATH_SEPARATOR;
		this.config = config;
	}

	/**
	 * 异步监测etcd恢复，并重试刷新配置项
	 * 5秒后开始，间隔5秒执行
	 */
	private void asyncRetry() {
		// 定时器，用于异步监测etcd的启动
		ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();
		executorService.scheduleAtFixedRate(new Runnable() {
			@Override
			public void run() {
				Thread.currentThread().setName("etcd-refresh");
				EtcdHealthResponse healthResponse = getSource().getHealth();
				ApplicationContext applicationContext = MainApplicationContextHolder.getApplicationContext();
				if ("true".equals(healthResponse.getHealth()) && applicationContext != null) {
					ContextRefresher contextRefresher = applicationContext.getBean(ContextRefresher.class);
					if (contextRefresher != null) {
						contextRefresher.refresh();
						executorService.shutdown();
					}
				}
			}
		}, 5, 5, TimeUnit.SECONDS);
	}

	public void init() {
		try {
			final EtcdKeysResponse response = getSource()
					.getDir(getName())
					.recursive()
					.timeout(config.getTimeout(), config.getTimeoutUnit())
					.setRetryPolicy(new RetryPolicy(0) {	// 0 表示立即重试
						@Override
						public boolean shouldRetry(ConnectionState connectionState) {
							// 重试次数1次
							if (connectionState.retryCount <= 1) {
								return true;
							}
							// 异步监测etcd恢复，一旦恢复，刷新配置项
							asyncRetry();
							return false;
						}
					}).send().get();

			if (response.node != null) {
				process(response.node);
			}
		}
		catch (EtcdException e) {
			if (e.errorCode == 100) {//key not found, no need to print stack trace
				logger.debug("Unable to init property source: " + getName() + ", " + e.getMessage());
			} else {
				logger.warn("Unable to init property source: " + getName(), e);
			}
		}
		catch (Exception e) {
			logger.warn("Unable to init property source: " + getName(), e);

		}
	}

	@Override
	public String[] getPropertyNames() {
		return properties.keySet().toArray(new String[0]);
	}

	@Override
	public Object getProperty(String name) {
		return properties.get(name);
	}

	// *************************************************************************
	//
	// *************************************************************************

	private void process(final EtcdKeysResponse.EtcdNode root) {
		if (!StringUtils.isEmpty(root.value)) {
			final String key = root.key.substring(this.prefix.length());

			properties.put(key.replace(EtcdConstants.PATH_SEPARATOR,
					EtcdConstants.PROPERTIES_SEPARATOR), root.value);
		}

		if (!CollectionUtils.isEmpty(root.nodes)) {
			for (EtcdKeysResponse.EtcdNode node : root.nodes) {
				process(node);
			}
		}
	}
}
