package cn.com.duibaboot.ext.autoconfigure.gclog;

import cn.com.duiba.wolf.utils.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.endpoint.AbstractEndpoint;

import java.io.*;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

public class GclogEndpoint extends AbstractEndpoint<Map<String, String>> {

    private static final Logger LOGGER = LoggerFactory.getLogger(GclogEndpoint.class);

    public static final String GCLOG_PATH = "/dev/shm";

    public GclogEndpoint() {
        super("gclog");
    }

    public Map<String, String> invoke() {
        File gcLogDir = new File(GCLOG_PATH);
        File[] gcFiles = gcLogDir.listFiles();
        if (gcFiles == null) {
            return Collections.emptyMap();
        }

        Map<String, String> gcFileMap = new HashMap<>();
        for (File f : gcFiles) {
            if (f.isDirectory()) {
                continue;
            }
            String startTime;
            String endTime;
            try {
                startTime = getStartTime(f);
                endTime = getEndTime(f);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            String key = String.format("%s-%s", startTime, endTime);
            gcFileMap.put(key, f.getAbsolutePath());
        }
        return gcFileMap;
    }

    /**
     * 获取gclog文件的开始时间
     * @param file
     * @return
     * @throws IOException
     */
    private String getStartTime(File file) throws IOException {
        FileInputStream fileInputStream = null;
        InputStreamReader inputStreamReader = null;
        BufferedReader bufferedReader = null;
        try {
            fileInputStream = new FileInputStream(file);
            inputStreamReader = new InputStreamReader(fileInputStream);
            bufferedReader = new BufferedReader(inputStreamReader);

            String line;
            while ((line = bufferedReader.readLine()) != null) {
                try {
                    return parseTime(line);
                } catch (ParseException e) {
                    LOGGER.info("非时间行。");
                }
            }
            throw new RuntimeException("gclog文件格式异常");
        } finally {
            if (fileInputStream != null) {
                fileInputStream.close();
            }
            if (inputStreamReader != null) {
                inputStreamReader.close();
            }
            if (bufferedReader != null) {
                bufferedReader.close();
            }
        }
    }

    /**
     * 获取gclog文件的结束时间
     * @param file
     * @return
     * @throws Exception
     */
    private String getEndTime(File file) throws Exception {
        RandomAccessFile rf = null;
        try {
            rf = new RandomAccessFile(file.getAbsolutePath(), "r");
            long len = rf.length();
            long start = rf.getFilePointer();
            long nextend = start + len - 1;
            String line = "";
            rf.seek(nextend);
            int c = -1;
            int t = 0;
            while (nextend > start) {
                c = rf.read();
                if (c == '\n' || c == '\r') {
                    line = rf.readLine();
                    t++;
                    if (t >= 1 && line != null) {
                        try {
                            return parseTime(line);
                        } catch (Exception e) {
                            LOGGER.info("非时间开头的行。");
                        }
                    }
                    nextend--;
                }
                nextend--;
                rf.seek(nextend);
                if (nextend == 0) {// 当文件指针退至文件开始处，解析第一行
                    try {
                        return parseTime(rf.readLine());
                    } catch (Exception e) {
                        LOGGER.info("非时间开头的行。");
                    }
                }
            }
            throw new RuntimeException("gclog文件格式异常");
        } finally {
            if (rf != null) {
                rf.close();
            }
        }
    }

    private String parseTime(String line) throws ParseException {
        String timestamp = line.substring(0, 19);
        Date date = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss").parse(timestamp);
        return DateUtils.getSecondStr(date);
    }

}
