package cn.com.duibaboot.ext.autoconfigure.threadpool;

import cn.com.duibaboot.ext.autoconfigure.threadpool.properties.DuibaThreadPoolConfig;
import cn.com.duibaboot.ext.autoconfigure.threadpool.properties.ScheduledThreadPoolProperties;
import cn.com.duibaboot.ext.autoconfigure.threadpool.wrapper.ScheduledThreadPoolExecutorWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.scheduling.annotation.EnableAsync;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 自动构建线程池，使用文档：http://cf.dui88.com/pages/viewpage.action?pageId=4505751
 * Created by wenqi.huang on 2017/2/7.
 */
@EnableAsync
@Configuration
@Import(ThreadPoolBeanFactoryPostProcessor.class)
@EnableConfigurationProperties({DuibaThreadPoolConfig.class})
public class ThreadPoolAutoConfiguration{

    private static Logger logger = LoggerFactory.getLogger(ThreadPoolAutoConfiguration.class);

    @Autowired
    private DuibaThreadPoolConfig duibaThreadPoolConfig;

    /**
     * 配置定时调度线程池,返回的线程池实现了SmartLifecycle，会在spring关闭的时候首先被关闭
     * @return
     */
    @Bean(name = ScheduledThreadPoolProperties.DEFAULT_BEAN_NAME, destroyMethod = "shutdown")
    @ConditionalOnProperty(name="duiba.threadpool.scheduled.enabled", havingValue="true", matchIfMissing = false)
    public ScheduledExecutorService scheduledExecutorService(){

        ScheduledThreadPoolProperties poolProperties = duibaThreadPoolConfig.getScheduled();

        return new ScheduledThreadPoolExecutorWrapper(poolProperties);
    }

    /**
     * 设置默认的UncaughtExceptionHandler，当线程运行报错时打印错误到slf4j中。
     * @return
     */
    @Configuration
    public static class DuibaUncaughtExceptionHandlerConfiguarApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

        private AtomicBoolean flag = new AtomicBoolean(true);

        @Override
        public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
            if(!flag.compareAndSet(true,false)){
                return;
            }

            if(Thread.getDefaultUncaughtExceptionHandler() == null){
                Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                    @Override
                    public void uncaughtException(Thread t, Throwable e) {
                        if(e instanceof ThreadDeath){
                            //捕获到这个异常表示当前线程希望安静地退出，满足它吧，这里就不打印任何日志了。
                        }
                        logger.error("Thread["+t.getId()+","+t.getName()+"] caught error:", e);
                    }
                });
            }
        }

        @Override
        public int getOrder() {
            return -15;
        }
    }

}
