package cn.com.duibaboot.ext.autoconfigure.web.container;

import cn.com.duibaboot.ext.autoconfigure.core.DuibaServerProperties;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.MonitorCallable;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.MonitorRunnable;
import org.apache.coyote.http11.Http11NioProtocol;
import org.apache.tomcat.util.net.AbstractEndpoint;
import org.apache.tomcat.util.threads.TaskQueue;
import org.apache.tomcat.util.threads.TaskThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.embedded.tomcat.TomcatEmbeddedServletContainerFactory;
import org.springframework.util.ReflectionUtils;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.*;

/**
 * 自动配置Tomcat等容器，增加http线程池监控，如果在队列中等待超过1秒，则打印error日志提示线程池过小<br/>
 * 如果是内部服务，使用固定大小的线程池,并拒绝在http线程池工作队列中等待过久的任务，让客户端可以尽快重试
 */
public class TomcatCustomizer extends SpecifiedEmbeddedServletContainerCustomizer<TomcatEmbeddedServletContainerFactory> {
    private static final Logger logger = LoggerFactory.getLogger(TomcatCustomizer.class);
    private static volatile DuibaServerProperties duibaServerProperties;

    private static volatile TomcatCustomizer INSTANCE;

    public TomcatCustomizer(DuibaServerProperties duibaServerProperties){
        super(TomcatEmbeddedServletContainerFactory.class);
        TomcatCustomizer.duibaServerProperties = duibaServerProperties;
        INSTANCE = this;
    }

    @Override
    public void customizeSpecified(TomcatEmbeddedServletContainerFactory container) {
        container.setProtocol(DuibaHttp11NioProtocol.class.getName());
    }

    @Override
    public void shutdownGracefully(){
    }

    public static class DuibaHttp11NioProtocol extends Http11NioProtocol{

        private org.apache.tomcat.util.threads.ThreadPoolExecutor http11NioExecutor;
        private long executorTerminationTimeoutMillis;
        private String endpointName;

        public DuibaHttp11NioProtocol(){
            super();
            //设置backlog参数,以防很多客户端同时连接时，有客户抛出connection timeout的问题
            super.setAcceptCount(1000);
        }

        public AbstractEndpoint getEndpointInner() throws InvocationTargetException, IllegalAccessException{
            try {
                //直接调用可能会报错(tomcat不同版本api变化较大导致)，作为降级，使用反射调用一次。
                return super.getEndpoint();
            }catch(Throwable e) {
                Class<?> clazz = Http11NioProtocol.class;
                Method endpointMethod = ReflectionUtils.findMethod(clazz, "getEndpoint");
                endpointMethod.setAccessible(true);
                AbstractEndpoint endpoint = (AbstractEndpoint) endpointMethod.invoke(this, null);
                return endpoint;
            }
        }

        private void createExecutor() throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
            AbstractEndpoint endpoint = getEndpointInner();
            executorTerminationTimeoutMillis = endpoint.getExecutorTerminationTimeoutMillis();
            endpointName = endpoint.getName();
            if (http11NioExecutor == null) {
                TaskThreadFactory tf = new TaskThreadFactory(endpointName + "-exec-", true, getThreadPriority());
                TomcatMonitorThreadPoolExecutor executor;
                int minThreads = getMinSpareThreads();
                long timeToLive = 60;
                if(duibaServerProperties.isInternalMode()){//如果是内部服务，使用固定大小的线程池
                    minThreads = getMaxThreads();
                    timeToLive = 0;
                }

                //使用这个taskQueue可以确保线程池会先使用到MaxPoolSize才开始入队
                TaskQueue taskqueue = new TaskQueue();
                executor = new TomcatMonitorThreadPoolExecutor(minThreads, getMaxThreads(), timeToLive, TimeUnit.SECONDS, taskqueue, tf);
                taskqueue.setParent( executor);

                http11NioExecutor = executor;
            }
        }

        @Override
        public void start() throws Exception {
            try {
                if(duibaServerProperties.isInternalMode()){
//                    getEndpointInner().setKeepAliveTimeout();
                    //对于每一个keep-alive的http长连接，如果请求次数达到此限制，server会强制断开连接;
                    //内部服务模式把此限制调大，以尽量减少断开重连的次数（默认100）
                    getEndpointInner().setMaxKeepAliveRequests(Integer.MAX_VALUE);
                }
                createExecutor();
                setExecutor(http11NioExecutor);

                //如果是内部服务，则开启异步扫描线程，扫描出所有在队列中等待过久的任务，直接拒绝，以达到failfast的目的.（配合FailFastFilter共同生效）
                if(duibaServerProperties.isInternalMode()){
                    TomcatCustomizer.INSTANCE.failFastToHttpExecutor(http11NioExecutor);
                }

            }catch(Exception e){
                logger.warn("getEndpoint 失败，不会给tomcat注入线程池监控功能", e);
            }

            super.start();
        }

        @Override
        public void stop() throws Exception {
            super.stop();

            if(http11NioExecutor != null){
                if (http11NioExecutor instanceof org.apache.tomcat.util.threads.ThreadPoolExecutor) {
                    //this is our internal one, so we need to shut it down
                    org.apache.tomcat.util.threads.ThreadPoolExecutor tpe = (org.apache.tomcat.util.threads.ThreadPoolExecutor) http11NioExecutor;
                    tpe.shutdownNow();
                    if (executorTerminationTimeoutMillis > 0) {
                        try {
                            tpe.awaitTermination(executorTerminationTimeoutMillis, TimeUnit.MILLISECONDS);
                        } catch (InterruptedException e) {
                            // Ignore
                        }
                        if (tpe.isTerminating()) {
                            getLog().warn(sm.getString("endpoint.warn.executorShutdown", endpointName));
                        }
                    }
                    TaskQueue queue = (TaskQueue) tpe.getQueue();
                    queue.setParent(null);
                }
            }
        }

    }

    protected static class TomcatMonitorThreadPoolExecutor extends org.apache.tomcat.util.threads.ThreadPoolExecutor {

        public TomcatMonitorThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, RejectedExecutionHandler handler) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, handler);
        }

        public TomcatMonitorThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
        }

        public TomcatMonitorThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory);
        }

        public TomcatMonitorThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue) {
            super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue);
        }

        @Override
        public void execute(Runnable command) {
            super.execute(new MonitorRunnable(command, getQueue()));
        }

        @Override
        public <T> Future<T> submit(Callable<T> task) {
            return super.submit(new MonitorCallable(task, getQueue()));
        }

        @Override
        public <T> Future<T> submit(Runnable task, T result) {
            return super.submit(new MonitorRunnable(task, getQueue()), result);
        }

        @Override
        public Future<?> submit(Runnable task) {
            return super.submit(new MonitorRunnable(task, getQueue()));
        }

    }

}
