package cn.com.duibaboot.ext.autoconfigure.cat;

import com.google.common.collect.ImmutableSet;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.hadoop.hbase.client.Table;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.sleuth.ErrorParser;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.stereotype.Component;

@Aspect
public class SleuthHbasePlugin {

    private static final String space = "Hbase";

    @Autowired
    private Tracer tracer;
    @Autowired
    private ErrorParser errorParser;

    private static final ImmutableSet<String> BLACK_METHODS = ImmutableSet.of("toString","hashCode","equals","getClass","wait","notify","notifyAll");
    private static final ImmutableSet<String> WHITE_METHODS = ImmutableSet.of("exists","existsAll","batch","batchCallback","get","getScanner","put",
        "checkAndPut","delete","checkAndDelete","mutateRow","append","increment","incrementColumnValue"
    );


    @Around("execution(* org.apache.hadoop.hbase.client.HTableInterfaceFactory.createHTableInterface(..))")
    public Object springDataMongodbJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {
        Table htable = (Table)joinPoint.proceed();
        ProxyFactory factory = new ProxyFactory();
        factory.setTarget(htable);
        factory.addAdvice(new HbaseMethodInterceptor());
        return factory.getProxy();
    }

    /**
     * {@link MethodInterceptor}
     */
    public class HbaseMethodInterceptor implements MethodInterceptor {

        HbaseMethodInterceptor() {
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            String methodName = invocation.getMethod().getName();
            if(BLACK_METHODS.contains(methodName) || !WHITE_METHODS.contains(methodName)){
                return invocation.proceed();
            }
            if(!tracer.isTracing() || !tracer.getCurrentSpan().isExportable()){
                return invocation.proceed();
            }

            Class<?> clazz = invocation.getThis().getClass();

            Span span = tracer.createSpan("Hbase:/"+methodName);
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append(clazz.getSimpleName()).append(".").append(methodName);
            try {
                span.tag("Hbase.class_method", stringBuilder.toString());// get/...
                span.tag(Span.SPAN_LOCAL_COMPONENT_TAG_NAME,space);//本地组件名
                span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME,space);//远程服务名
                span.logEvent(Span.CLIENT_SEND);

                return invocation.proceed();
            } catch(Exception e){
                errorParser.parseErrorTags(span, e);
                throw e;
            } finally {
                span.logEvent(Span.CLIENT_RECV);
                tracer.close(span);
            }
        }

    }

}
