package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.context.embedded.EmbeddedServletContainerInitializedEvent;
import org.springframework.cloud.netflix.eureka.serviceregistry.EurekaAutoServiceRegistration;
import org.springframework.context.event.EventListener;

import java.lang.reflect.Method;

/**
 * aop强制延迟注册eureka，延迟到web端口绑定后再注册到eureka。
 * <br/>
 * 确保在web容器启动完成前，eureka状态不会改为UP, 以避免kong检测到UP状态的服务器直接放行流量导致访问不通的问题。
 *
 */
public class EurekaAutoServiceRegistrationPostProcessor implements SpecifiedBeanPostProcessor<EurekaAutoServiceRegistration> {

    private static final Logger logger = LoggerFactory.getLogger(EurekaAutoServiceRegistrationPostProcessor.class);

    @Override
    public Class<EurekaAutoServiceRegistration> getBeanType() {
        return EurekaAutoServiceRegistration.class;
    }

    @Override
    public Object postProcessBeforeInitialization(EurekaAutoServiceRegistration bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(EurekaAutoServiceRegistration bean, String beanName) throws BeansException {
        boolean canAop = true;
        try {
            Class.forName("org.springframework.boot.context.embedded.EmbeddedServletContainerInitializedEvent");
            Class.forName("javax.servlet.Servlet");
            Method method = EurekaAutoServiceRegistration.class.getDeclaredMethod("onApplicationEvent", EmbeddedServletContainerInitializedEvent.class);
            EventListener el = method.getAnnotation(EventListener.class);
            if(el == null || el.value() == null || el.value().length != 1 || !el.value()[0].equals(EmbeddedServletContainerInitializedEvent.class)){
                canAop = false;
            }
        }catch(NoSuchMethodException | ClassNotFoundException e){
            canAop = false;
        }
        if(!canAop){
            //如果不在web环境下或者EurekaAutoServiceRegistration没有监听EmbeddedServletContainerInitializedEvent的方法，则不做代理
            logger.warn("检测到没有启用web container，不会自动启用延迟注册eureka特性");
            return bean;
        }
        //aop强制延迟注册eureka，延迟到web端口绑定后再注册到eureka。
        ProxyFactory proxy = new ProxyFactory();
        proxy.setTarget(bean);
        proxy.addAdvice(new EurekaAutoServiceRegistrationMethodInterceptor());
        return proxy.getProxy();
    }

    @Override
    public int getOrder() {
        return 0;
    }

    private static class EurekaAutoServiceRegistrationMethodInterceptor implements MethodInterceptor {

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(invocation.getMethod().getName().equals("isAutoStartup")){//设置不自动start
                return false;
            }

            return invocation.proceed();
        }
    }
}
