package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import feign.Feign;
import feign.MethodMetadata;
import feign.Param;
import org.springframework.cloud.netflix.feign.AnnotatedParameterProcessor;
import org.springframework.cloud.netflix.feign.support.SpringMvcContract;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.HttpHeaders;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;

import static feign.Util.checkState;
import static feign.Util.emptyToNull;
import static org.springframework.core.annotation.AnnotatedElementUtils.findMergedAnnotation;

/**
 *使用自定义的 Contract，对于AdvancedFeignClient进行特殊处理，自动扫描AdvancedFeignClient下的所有方法为FeignClient(使用方法名作为path)，并把所有参数使用json序列化
 */
public class CustomSpringMvcContract extends SpringMvcContract {

    private static final String ACCEPT = "Accept";
    private static final String CONTENT_TYPE = "Content-Type";
    /**
     *AdvancedFeignClient模式下会自动按mvc方法的参数顺序来组装参数名字，比如第一个参数为_p0,第二个为_p1，依此类推
     */
    public static final String HTTP_PARAMETER_PREFIX = "_p";
    public static final String HTTP_HEADER_PARAM_COLLECTION_SPLITTED = "Param-Collection-Splitted";

    private DuibaFeignProperties duibaFeignProperties;

    private ResourceLoader resourceLoader = new DefaultResourceLoader();
    private final Map<String, Method> processedMethods = new HashMap<>();
    private final ConversionService object2jsonStringConversionService;
    private final Param.Expander object2jsonStringExpander;

    {
        GenericConversionService cs = new GenericConversionService();
        cs.addConverter(new ObjectToJsonStringConverter());
        object2jsonStringConversionService = cs;
        object2jsonStringExpander = new ConvertingExpander(object2jsonStringConversionService);
    }

    public CustomSpringMvcContract(){
        super();
    }

    public CustomSpringMvcContract(List<AnnotatedParameterProcessor> annotatedParameterProcessors) {
        super(annotatedParameterProcessors);
    }

    public CustomSpringMvcContract(List<AnnotatedParameterProcessor> annotatedParameterProcessors, ConversionService conversionService, DuibaFeignProperties duibaFeignProperties) {
        super(annotatedParameterProcessors, conversionService);
        this.duibaFeignProperties = duibaFeignProperties;
    }

    @Override
    public MethodMetadata parseAndValidateMetadata(Class<?> targetType, Method method) {
        this.processedMethods.put(Feign.configKey(targetType, method), method);

        return super.parseAndValidateMetadata(targetType, method);
    }

    @Override
    protected void processAnnotationOnClass(MethodMetadata data, Class<?> clz) {
        super.processAnnotationOnClass(data, clz);

        if (clz.isAnnotationPresent(AdvancedFeignClient.class)){
            //如果类上注解了@AdvancedFeignClient则进行特殊处理, 无论方法上有没有RequestMapping注解，都扫描为FeignClient方法
            Method method = processedMethods.get(data.configKey());

            //url为空表示类上没有RequestMappping注解或者有注解但是value或path为空，如果是这种情况，则使用类名作为path(类名首字母小写)
            if(!StringUtils.hasText(data.template().url())){
                String pathValue = StringUtils.uncapitalize(clz.getSimpleName());
                data.template().insert(0, pathValue);
            }

            processAdvancedFeignClientMethod(data, method);
        }
    }

    @Override
    protected void processAnnotationOnMethod(MethodMetadata data,
                                             Annotation methodAnnotation, Method method) {
        if (method.getDeclaringClass().isAnnotationPresent(AdvancedFeignClient.class)){
            //如果类上注解了@AdvancedFeignClient则无论有没有RequestMapping注解，都扫描为FeignClient方法并进行特殊处理

            //已经在processAnnotationOnClass中提前处理了，这里不再处理,直接返回
            return;
        }

        super.processAnnotationOnMethod(data, methodAnnotation, method);
    }

    @Override
    protected boolean processAnnotationsOnParameter(MethodMetadata data,
                                                    Annotation[] annotations, int paramIndex) {
        Method method = processedMethods.get(data.configKey());
        if (method.getDeclaringClass().isAnnotationPresent(AdvancedFeignClient.class)){
            //如果类上注解了@AdvancedFeignClient则进行特殊处理, 返回true，表示isHttpAnnotation

            if(duibaFeignProperties.getSerializationEnum() != DuibaFeignProperties.DuibaFeignSerialization.JSON){
                return true;
            }

            String parameterName = HTTP_PARAMETER_PREFIX + paramIndex;//第一个参数_p0,第二个_p1，以此类推
            List<String> query = new ArrayList<>();
            query.add(String.format("{%s}", parameterName));
            data.template().query(parameterName, query);
            Collection<String>
                    names =
                    data.indexToName().containsKey(paramIndex) ? data.indexToName().get(paramIndex) : new ArrayList<>();
            names.add(parameterName);
            data.indexToName().put(paramIndex, names);

            if (data.indexToExpander().get(paramIndex) == null
                    //&& !isMultiValued(method.getParameterTypes()[paramIndex])
                    && this.object2jsonStringConversionService.canConvert(
                    method.getParameterTypes()[paramIndex], String.class)) {
                //使用Object2JsonString转换器
                data.indexToExpander().put(paramIndex, this.object2jsonStringExpander);
            }

            return true;
        }

        return super.processAnnotationsOnParameter(data, annotations, paramIndex);
    }
//
//    private boolean isMultiValued(Class<?> type) {
//        // Feign will deal with each element in a collection individually (with no
//        // expander as of 8.16.2, but we'd rather have no conversion than convert a
//        // collection to a String (which ends up being a csv).
//        return Collection.class.isAssignableFrom(type);
//    }

    /**
     * 自动识别出方法名做为path
     * @param data
     * @param method
     */
    private void processAdvancedFeignClientMethod(MethodMetadata data, Method method){
        RequestMapping methodMapping = findMergedAnnotation(method, RequestMapping.class);
        // HTTP Method
        RequestMethod[] methods = methodMapping == null ? new RequestMethod[] { RequestMethod.POST } : methodMapping.method();
        if (methods.length == 0) {
            methods = new RequestMethod[] { RequestMethod.POST };
        }
        checkOne(method, methods, "method");
        data.template().method(methods[0].name());

        // path
        if(methodMapping != null) {
            checkAtMostOne(method, methodMapping.value(), "value");
            if (methodMapping.value().length > 0) {
                String pathValue = emptyToNull(methodMapping.value()[0]);
                if (pathValue != null) {
                    pathValue = resolve(pathValue);
                    // Append path from @RequestMapping if value is present on method
                    if (!pathValue.startsWith("/")
                            && !data.template().toString().endsWith("/")) {
                        pathValue = "/" + pathValue;
                    }
                    data.template().append(pathValue);
                }else{
                    //使用方法名作为path
                    useMethodNameAsPath(data, method);
                }
            }else{
                //使用方法名作为path
                useMethodNameAsPath(data, method);
            }
            // produces
            parseProduces(data, method, methodMapping);
            // consumes
            parseConsumes(data, method, methodMapping);
            // headers
            parseHeaders(data, method, methodMapping);
        }else{
            //使用方法名作为path
            useMethodNameAsPath(data, method);
        }

        data.indexToExpander(new LinkedHashMap<>());

        //需要附加特殊http头:Param-Collection-Splitted=1 来表示Collection参数会被自动分割(FeignClient处理策略)
        if(duibaFeignProperties.getSerializationEnum() == DuibaFeignProperties.DuibaFeignSerialization.JSON) {
            data.template().header(HTTP_HEADER_PARAM_COLLECTION_SPLITTED, "1");
        } else {
            //X-SERIAL头要求服务端反序列化请求体时用指定的序列化方式反序列化
            data.template().header(CustomRequestInterceptor.X_SERIAL, duibaFeignProperties.getSerializationEnum().getType());
            //Accept头要求服务端响应时也以指定的序列化方式返回数据
            data.template().header(HttpHeaders.ACCEPT, duibaFeignProperties.getSerializationEnum().getContentType());
        }
    }

    /**
     * 使用方法名作为path
     */
    private void useMethodNameAsPath(MethodMetadata data, Method method){
        String pathValue = method.getName();
        if (!data.template().toString().endsWith("/")) {
            pathValue = "/" + pathValue;
        }
        data.template().append(pathValue);
    }

    private void checkOne(Method method, Object[] values, String fieldName) {
        checkState(values != null && values.length == 1,
                "Method %s can only contain 1 %s field. Found: %s", method.getName(),
                fieldName, values == null ? null : Arrays.asList(values));
    }

    private void checkAtMostOne(Method method, Object[] values, String fieldName) {
        checkState(values != null && (values.length == 0 || values.length == 1),
                "Method %s can only contain at most 1 %s field. Found: %s",
                method.getName(), fieldName,
                values == null ? null : Arrays.asList(values));
    }

    private String resolve(String value) {
        if (StringUtils.hasText(value)
                && this.resourceLoader instanceof ConfigurableApplicationContext) {
            return ((ConfigurableApplicationContext) this.resourceLoader).getEnvironment()
                    .resolvePlaceholders(value);
        }
        return value;
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        super.setResourceLoader(resourceLoader);
        this.resourceLoader = resourceLoader;
    }

    private void parseProduces(MethodMetadata md, Method method,
                               RequestMapping annotation) {
        checkAtMostOne(method, annotation.produces(), "produces");
        String[] serverProduces = annotation.produces();
        String clientAccepts = serverProduces.length == 0 ? null
                : emptyToNull(serverProduces[0]);
        if (clientAccepts != null) {
            md.template().header(ACCEPT, clientAccepts);
        }
    }

    private void parseConsumes(MethodMetadata md, Method method,
                               RequestMapping annotation) {
        checkAtMostOne(method, annotation.consumes(), "consumes");
        String[] serverConsumes = annotation.consumes();
        String clientProduces = serverConsumes.length == 0 ? null
                : emptyToNull(serverConsumes[0]);
        if (clientProduces != null) {
            md.template().header(CONTENT_TYPE, clientProduces);
        }
    }

    private void parseHeaders(MethodMetadata md, Method method,
                              RequestMapping annotation) {
        // TODO: only supports one header value per key
        if (annotation.headers() != null && annotation.headers().length > 0) {
            for (String header : annotation.headers()) {
                int index = header.indexOf('=');
                md.template().header(resolve(header.substring(0, index)),
                        resolve(header.substring(index + 1).trim()));
            }
        }
    }
}
