package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.hystrix;

import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixCommand;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixProperty;
import cn.com.duiba.boot.netflix.feign.hystrix.conf.HystrixPropertiesManager;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.etcd.config.EtcdConstants;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.hystrix.*;
import feign.Feign;
import feign.RetryableException;
import feign.Retryer;
import feign.Target;
import feign.codec.ErrorDecoder;
import feign.hystrix.HystrixFeign;
import feign.hystrix.SetterFactory;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.apache.http.conn.ConnectTimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cloud.netflix.feign.CustomFeignClientsRegistrar;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;
import org.springframework.core.env.Environment;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.SocketTimeoutException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.RejectedExecutionException;

/**
 * Feign客户端自定义，覆盖 FeignClientsConfiguration 中对应配置
 */
@Configuration
@ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
public class HystrixFeignConfiguration  {
	@Autowired
	private Environment environment;

	private static Map<String,Object> circuitBreakerPropety = new ConcurrentHashMap<>();

    private static final Logger logger = LoggerFactory.getLogger(HystrixFeignConfiguration.class);
	//hystrix 线程池相关配置
	private static final Set<String> hystrixThreadPoolKeyNames = Sets.newHashSet(HystrixPropertiesManager.MAX_QUEUE_SIZE,
			HystrixPropertiesManager.CORE_SIZE,
			HystrixPropertiesManager.MAXIMUM_SIZE,
			HystrixPropertiesManager.ALLOW_MAXIMUM_SIZE_TO_DIVERGE_FROM_CORE_SIZE,
			HystrixPropertiesManager.KEEP_ALIVE_TIME_MINUTES,
			HystrixPropertiesManager.QUEUE_SIZE_REJECTION_THRESHOLD,
			HystrixPropertiesManager.METRICS_ROLLING_STATS_NUM_BUCKETS,
			HystrixPropertiesManager.METRICS_ROLLING_STATS_TIME_IN_MILLISECONDS
			);

	/**
	 * 使用自定义的SetterFactory,以获得在代码中自定义超时时间、自定义熔断器配置的特性
	 * @return
	 */
	@Bean
	@Scope("prototype")
	@ConditionalOnMissingBean
	@ConditionalOnProperty(name = "feign.hystrix.enabled", matchIfMissing = true)
	public Feign.Builder feignHystrixBuilder() {
		return HystrixFeign.builder().setterFactory(new HystrixFeignSetterFactory(environment.getProperty(EtcdConstants.PROPERTY_SPRING_APPLICATION_NAME)));
	}
    @Bean
    @ConditionalOnMissingBean
    public CircuitBreakerEndpoint circuitBreakerEndpoint() {
        CircuitBreakerEndpoint endpoint = new CircuitBreakerEndpoint(circuitBreakerPropety);
        return endpoint;
    }

	//如果引入了sleuth依赖，则上述的feignHystrixBuilder方法不会被调用，所以需要加入这个BeanPostProcessor。
	@Bean
	public static SpecifiedBeanPostProcessor<Feign.Builder> feignBuildPostProcessor(Environment environment){
		return new SpecifiedBeanPostProcessor<Feign.Builder>() {
			@Override
			public Class<Feign.Builder> getBeanType() {
				return Feign.Builder.class;
			}

			@Override
			public Object postProcessBeforeInitialization(Feign.Builder bean, String beanName) throws BeansException {
				return bean;
			}

			@Override
			public Object postProcessAfterInitialization(Feign.Builder bean, String beanName) throws BeansException {
				if(bean instanceof feign.hystrix.HystrixFeign.Builder){
					((feign.hystrix.HystrixFeign.Builder)bean).setterFactory(new HystrixFeignSetterFactory(environment.getProperty(EtcdConstants.PROPERTY_SPRING_APPLICATION_NAME)));
				}
				return bean;
			}

			@Override
			public int getOrder() {
				return 0;
			}
		};
	}

	/**
	 * 自定义Feign错误解码器，把非2**状态的http响应转换为异常。对于IllegalArgumentException或者注解了@HystrixDisabled的异常会包装为HystrixBadRequestException，避免触发熔断/降级
	 * @return
	 */
	@Bean
	public ErrorDecoder feignErrorDecoder(){
		return new FeignErrorDecoder();
	}

	/**
	 * 自定义重试处理器,遇到RejectedExecutionException进行重试。(这里不处理ConnectException，因为Ribbon内部的重试处理器会对这个进行处理进行重试, 为了避免重复处理，这里不需要处理ConnectionException）
	 */
	@Configuration
	public static class CustomRetryer implements Retryer {
		private int maxAttempts = 2;
		private int attempt = 1;

		@Override
		public void continueOrPropagate(RetryableException e) {
			if(attempt == 1 && RpcContext.hasContext()){
				//给异常信息附加rpc的本地ip端口信息和远程ip端口信息，方便排错
				RpcContext rc = RpcContext.getContext();
				Field msgField = FieldUtils.getDeclaredField(Throwable.class, "detailMessage", true);
				try {
					FieldUtils.writeField(msgField, e, e.getMessage() + " (" + rc.getLocalAddr() + " -> " + rc.getRemoteAddr() + ")");
				} catch (IllegalAccessException e1) {
					throw new RuntimeException("[NOTIFYME] will never be here", e1);
				}
			}

			if(attempt++ >= maxAttempts){
				throw e;
			}
			Throwable cause = e.getCause();
			if(cause == null) {
				throw e;
			}
			else if(cause instanceof RejectedExecutionException){
				//RejectedExecutionException表示服务端线程池满了或者繁忙或者其他原因拒绝,50ms后重试1次
				try {
					Thread.sleep(50);
				} catch (InterruptedException e1) {
					Thread.currentThread().interrupt();
				}
			}else if(isConnectException(cause)){
			    //java Socket 连接超时错误：java.net.SocketTimeoutException: connect timed out
                // 遇到此异常重试一次.
                String msg = "connect timed out";
                if(cause instanceof ConnectTimeoutException){
                    msg = cause.getMessage();//Connect to www.baidu.com:80 failed: connect timed out
                }
                logger.warn(msg + ", retry again. reqId:{}", "undefined");
                try {
                    Thread.sleep(10);
                } catch (InterruptedException e1) {
                    Thread.currentThread().interrupt();
                }
            }else{
				throw e;
			}
		}

		private boolean isConnectException(Throwable cause){
			if(cause instanceof ConnectTimeoutException){
				return true;
			}
			if(cause instanceof SocketTimeoutException && "connect timed out".equals(cause.getMessage())){
				return true;
			}

			return false;
		}

		@Override
		public Retryer clone() {
			return new CustomRetryer();
		}
	}

	/**
	 * 这个类会自动识别出FeignClient(类、方法)上注解了@FeignHystrixProperty/@FeignHystrixCommand的方法，并应用其中的属性，比如超时时间等
	 */
	static final class HystrixFeignSetterFactory implements SetterFactory {

		private String currentApplicationName;

		public HystrixFeignSetterFactory(String currentApplicationName){
			this.currentApplicationName = currentApplicationName;
		}

		@Override
		public HystrixCommand.Setter create(Target<?> target, Method method) {
			String currentAppKey = currentApplicationName;
			String defaultGroupKey = target.name();
			String defaultCommandKey = Feign.configKey(target.type(), method);

			FeignHystrixCommand methodCommand = method.getAnnotation(FeignHystrixCommand.class);
			FeignHystrixProperty[] methodProperties = method.getAnnotationsByType(FeignHystrixProperty.class);
			FeignHystrixCommand classCommand = target.type().getAnnotation(FeignHystrixCommand.class);
			FeignHystrixProperty[] classProperties = target.type().getAnnotationsByType(FeignHystrixProperty.class);
			List<FeignHystrixCommand> commands = new ArrayList<>();
			commands.add(methodCommand);
			commands.add(classCommand);
			String groupKey = commands.stream().filter((c) -> c!=null && StringUtils.isNotBlank(c.groupKey())).findFirst().map(c -> c.groupKey()).orElse(defaultGroupKey);
			String commandKey = methodCommand == null || StringUtils.isBlank(methodCommand.commandKey()) ? defaultCommandKey : methodCommand.commandKey();
			String threadPoolKey = commands.stream().filter((c) -> c!=null && StringUtils.isNotBlank(c.threadPoolKey())).findFirst().map(c -> c.threadPoolKey()).orElse(null);

			HystrixCommand.Setter setter = HystrixCommand.Setter.withGroupKey(HystrixCommandGroupKey.Factory.asKey(groupKey));
			setter.andCommandKey(HystrixCommandKey.Factory.asKey(commandKey));
			if(threadPoolKey != null) {
				setter.andThreadPoolKey(HystrixThreadPoolKey.Factory.asKey(threadPoolKey));
			}

			//构造属性list，最低优先级的放在最前，以让后面的可以覆盖前面的配置
			List<FeignHystrixProperty> allProperties = new ArrayList<>();

			//从package-info中获取FeignHystrixProperty注解
			List<FeignHystrixProperty> packageProperties = new ArrayList<>();
			injectPropertiesFromPackages(target.type().getPackage().getName(), packageProperties);
			packageProperties = Lists.reverse(packageProperties);//反转过来，离当前类越近的包的配置优先级越高
			allProperties.addAll(packageProperties);

			List<FeignHystrixProperty> classAndMethodProperties = new ArrayList<>();

			if(classCommand != null && !ArrayUtils.isEmpty(classCommand.properties())){
				for(FeignHystrixProperty p : classCommand.properties()) {
					classAndMethodProperties.add(p);
				}
			}
			if(!ArrayUtils.isEmpty(classProperties)){
				for(FeignHystrixProperty p : classProperties) {
					classAndMethodProperties.add(p);
				}
			}
			if(methodCommand != null && !ArrayUtils.isEmpty(methodCommand.properties())){
				for(FeignHystrixProperty p : methodCommand.properties()) {
					classAndMethodProperties.add(p);
				}
			}
			if(!ArrayUtils.isEmpty(methodProperties)){
				for(FeignHystrixProperty p : methodProperties) {
					classAndMethodProperties.add(p);
				}
			}

			if((threadPoolKey == null && StringUtils.equals(defaultGroupKey, groupKey))
					|| (threadPoolKey != null && StringUtils.equals(threadPoolKey, defaultGroupKey))){
				//使用了默认的threadPoolKey时（groupKey改变时，threadPoolKey如果不指定则使用groupKey作为threadPoolKey），不允许指定线程池属性等HystrixThreadPoolKey级别的属性（参考https://github.com/Netflix/Hystrix/wiki/Configuration#execution.isolation.thread.timeoutInMilliseconds）
				validateHystrixThreadPoolKeyPropertiesNotExists(classAndMethodProperties, target.type(), method);
			}

			allProperties.addAll(classAndMethodProperties);
			HystrixCommandProperties.Setter commandProperties = HystrixPropertiesManager.initializeCommandProperties(allProperties);
			HystrixThreadPoolProperties.Setter threadPoolProperties = HystrixPropertiesManager.initializeThreadPoolProperties(allProperties);
			try {
				String commandFullPath = currentAppKey + EtcdConstants.PATH_SEPARATOR + (Strings.isNullOrEmpty(threadPoolKey)?groupKey:threadPoolKey) + EtcdConstants.PATH_SEPARATOR + commandKey;//#是特殊符号在etcd管理器无法正常访问
				publishCircuitBreakerProperty(circuitBreakerPropety, Splitter.on("/").splitToList(commandFullPath));
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			}
			setter.andCommandPropertiesDefaults(commandProperties);
			setter.andThreadPoolPropertiesDefaults(threadPoolProperties);
			return setter;
		}

		private void publishCircuitBreakerProperty(Map<String, Object> kv, List<String> propertyList) {

            String key = propertyList.get(0);
            if (!kv.containsKey(key)) {
				kv.put(key, new ConcurrentHashMap<>());
			}
            if (propertyList.size() > 1) {
				publishCircuitBreakerProperty((Map<String, Object>) kv.get(key), propertyList.subList(1, propertyList.size()));
            }
        }
		private void validateHystrixThreadPoolKeyPropertiesNotExists(List<FeignHystrixProperty> classAndMethodProperties, Class<?> clazz, Method method){
			for(FeignHystrixProperty p : classAndMethodProperties){
				if(hystrixThreadPoolKeyNames.contains(p.name())){
					throw new IllegalStateException(String.format("如果方法或类上没有指定非默认的groupKey/threadPoolKey（可以使用@FeignHystrixCommand注解指定），则你不能指定Hystrix线程池相关的配置，但是你指定了`%s`, 请检查类:`%s`, 方法: `%s` (警告：对于Hystrix线程池配置，hystrix会始终使用首次加载的配置，如果你在多个类或多个方法上对于同一个threadPoolKey指定了不同的线程池配置，则哪个配置生效是不确定的.)", p.name(), clazz.getName(), method.getName()));
				}
			}
		}

		private void injectPropertiesFromPackages(String packageName, List<FeignHystrixProperty> packageProperties){
			if(packageName == null){
				return;
			}
			try {
				//如果还没有加载过某个包下的类，则Package肯定找不到，所以这里先初始化一下
				Class.forName(packageName + ".package-info");
			} catch (ClassNotFoundException e) {
				//Ignore
			}
			Package pkg = Package.getPackage(packageName);
			if(pkg == null){
				//递归地到上级package寻找
				injectPropertiesFromPackages(parentPackage(packageName), packageProperties);
				return;
			}
			FeignHystrixProperty[] properties = pkg.getAnnotationsByType(FeignHystrixProperty.class);
			if(!ArrayUtils.isEmpty(properties)){
				int packageLevel = org.apache.commons.lang3.StringUtils.split(packageName, ".").length;
				if(packageLevel < CustomFeignClientsRegistrar.ADVANCED_FEIGN_CLIENT_ANNO_MIN_PACKAGE_LEVEL){
					//AdvancedFeignClient如果标注在package上，则这个package必须至少有三个层级,比如com.tuia.activitycenter,就是说只要要在项目级别，每个项目的package都不一样，以避免冲突.
					throw new IllegalStateException(String.format("@FeignHystrixProperty must be annotated on a package have at least %d levels, current package is : %s, level is: %d", CustomFeignClientsRegistrar.ADVANCED_FEIGN_CLIENT_ANNO_MIN_PACKAGE_LEVEL, packageName, packageLevel));
				}

				packageProperties.addAll(Arrays.asList(properties));
			}
			//递归地到上级package寻找
			injectPropertiesFromPackages(parentPackage(packageName), packageProperties);
		}

		private String parentPackage(String packageName){
			int idx = packageName.lastIndexOf(".");
			if(idx == -1){
				return null;
			}
			return packageName.substring(0, idx);
		}

	}

}
