package cn.com.duibaboot.ext.autoconfigure.etcd.client;

import cn.com.duibaboot.ext.autoconfigure.etcd.exception.EtcdOperationException;
import com.coreos.jetcd.Client;
import com.coreos.jetcd.KV;
import com.coreos.jetcd.data.ByteSequence;
import com.coreos.jetcd.data.KeyValue;
import com.coreos.jetcd.kv.DeleteResponse;
import com.coreos.jetcd.kv.GetResponse;
import com.coreos.jetcd.kv.PutResponse;
import com.coreos.jetcd.options.DeleteOption;
import com.coreos.jetcd.options.GetOption;
import org.apache.commons.collections.CollectionUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import static com.coreos.jetcd.data.ByteSequence.fromString;

/**
 * 封装jetcd的kvClient 提供etcd简单的 put get getWithPrefix delete deleteWithPrefix 操作
 * Created by guoyanfei .
 * 2018/7/12 .
 */
public class EtcdKVClientDelegate {

    private static final int      timeout     = 5;
    private static final TimeUnit timeoutUnit = TimeUnit.SECONDS;

    private final KV kvClient;

    public EtcdKVClientDelegate(Client etcdClient) {
        this.kvClient = etcdClient.getKVClient();
    }

    public void put(String key, String value) {
        CompletableFuture<PutResponse> putResponseCompletableFuture = this.kvClient.put(fromString(key), fromString(value));
        try {
            putResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
    }

    public String get(String key) {
        List<KeyValue> kvs = this.get(fromString(key), GetOption.DEFAULT);
        if (CollectionUtils.isEmpty(kvs)) {
            return null;
        }
        return kvs.get(0).getValue().toStringUtf8();
    }

    public Map<String, String> getWithPrefix(String prefix) {
        GetOption option = GetOption.newBuilder().withPrefix(fromString(prefix)).build();
        List<KeyValue> kvs = this.get(fromString(prefix), option);
        if (CollectionUtils.isEmpty(kvs)) {
            return Collections.emptyMap();
        }
        Map<String, String> resultMap = new HashMap<>();
        for (KeyValue kv : kvs) {
            resultMap.put(kv.getKey().toStringUtf8(), kv.getValue().toStringUtf8());
        }
        return resultMap;
    }

    private List<KeyValue> get(ByteSequence key, GetOption option) {
        CompletableFuture<GetResponse> getResponseCompletableFuture = this.kvClient.get(key, option);
        GetResponse getResponse;
        try {
            getResponse = getResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
        if (getResponse == null) {
            return Collections.emptyList();
        }
        return getResponse.getKvs();
    }

    public void delete(String key) {
        this.delete(fromString(key), DeleteOption.DEFAULT);
    }

    public void deleteWithPrefix(String prefix) {
        DeleteOption option = DeleteOption.newBuilder().withPrefix(fromString(prefix)).build();
        this.delete(fromString(prefix), option);
    }

    private void delete(ByteSequence key, DeleteOption option) {
        CompletableFuture<DeleteResponse> deleteResponseCompletableFuture = this.kvClient.delete(key, option);
        try {
            deleteResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
    }
}
