/*
 * Copyright 2013-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.etcd.config;

import cn.com.duibaboot.ext.autoconfigure.etcd.EtcdLastRefreshStateEndpoint;
import com.coreos.jetcd.Client;
import com.coreos.jetcd.data.KeyValue;
import com.coreos.jetcd.kv.GetResponse;
import com.coreos.jetcd.options.GetOption;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.EnumerablePropertySource;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.coreos.jetcd.data.ByteSequence.fromString;

/**
 * @author Luca Burgazzoli
 * @author Spencer Gibb
 */
public class EtcdPropertySource extends EnumerablePropertySource<Client> {

	private static final Logger logger = LoggerFactory.getLogger(EtcdPropertySource.class);

	private Map<String, String> properties;
	private final String prefix;
	private final EtcdConfigProperties config;

	private static volatile Map<String, String> lastValidProperties = new HashMap<>();

	public EtcdPropertySource(String root, Client source, EtcdConfigProperties config) {
		super(root, source);
		this.properties = new HashMap<>();
		this.prefix = root.startsWith(EtcdConstants.PATH_SEPARATOR) ? root
				+ EtcdConstants.PATH_SEPARATOR : EtcdConstants.PATH_SEPARATOR + root
				+ EtcdConstants.PATH_SEPARATOR;
		this.config = config;
	}

	public void init() {
		try {
			GetOption option = GetOption.newBuilder().withPrefix(fromString(prefix)).build();
			GetResponse response = getSource().getKVClient()
											  .get(fromString(prefix), option)
											  .get(config.getTimeout(), config.getTimeoutUnit());
			if (response.getCount() > 0) {
				process(response.getKvs());
			}
			EtcdLastRefreshStateEndpoint.markSuccess();
		} catch (Exception e) {
			logger.warn("Unable to init property source: " + getName(), e);
			EtcdLastRefreshStateEndpoint.markFail();
		}
		if (properties.size() == 0) {
			properties = lastValidProperties;
		} else {
			lastValidProperties = properties;
		}
	}

	@Override
	public String[] getPropertyNames() {
		return properties.keySet().toArray(new String[0]);
	}

	@Override
	public Object getProperty(String name) {
		return properties.get(name);
	}

	private void process(final List<KeyValue> kvs) {
	    if (CollectionUtils.isEmpty(kvs)) {
	        return;
        }
	    for (KeyValue kv : kvs) {
            String key = kv.getKey().toStringUtf8();
            String value = kv.getValue().toStringUtf8();
            key = key.substring(this.prefix.length());
            properties.put(key.replace(EtcdConstants.PATH_SEPARATOR, EtcdConstants.PROPERTIES_SEPARATOR), value);
        }
	}
}
