package cn.com.duibaboot.ext.autoconfigure.perftest.filter;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duibaboot.ext.autoconfigure.web.container.EmbeddedServletContainerThreadPoolHolder;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Resource;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 压测用的Filter,当从request识别到是压测请求时，设置context。
 * <br/>
 * 识别到servlet容器线程池繁忙时直接拒绝当前压测请求，如果5秒内拒绝了10个以上压测请求则直接熔断5秒。
 */
@Slf4j
public class PerfTestFilter implements Filter {

    @Resource
    private PerfTestRejectProperties perfTestRejectProperties;

    private LoadingCache<String, RejectedCountDto> servicePerfTestRejectedRequestCountCache = Caffeine.newBuilder()
            .expireAfterWrite(5, TimeUnit.SECONDS)
            .initialCapacity(1)
            .maximumSize(1)
            .build(s -> new RejectedCountDto());

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        PerfTestContext._setPerfTestMode(false);

        boolean isPerfTestMode = PerfTestUtils.isPerfTestRequest((HttpServletRequest)request);

        if(isPerfTestMode) {
            PerfTestContext._setPerfTestMode(true);
            if(isServerBusy()){
                HttpServletResponse resp = (HttpServletResponse)response;
                resp.setStatus(429);//429 too many requests.
                resp.getWriter().write("server busy(_duibaPerf=true)");//只有压测时才会显示这个信息
                return;
            }
        }
        try {
            chain.doFilter(request, response);
        } finally {
            PerfTestContext._setPerfTestMode(false);
        }
    }

    private boolean isServerBusy(){
        ThreadPoolExecutor es = EmbeddedServletContainerThreadPoolHolder.getServletContainerThreadPool();
        if(es == null){
            log.warn("获取不到servlet容器线程池，无法判读是否要对压测请求进行拒绝或熔断，请确认");
            return false;
        }

        if(isCircuitBreak()){
            return true;
        }

        int activeCount = es.getActiveCount();
        int coreSize = es.getCorePoolSize();//核心线程数配置
        int poolSize = es.getPoolSize();//当前池中实际分配的线程数
        if(EmbeddedServletContainerThreadPoolHolder.getServletContainerType() == EmbeddedServletContainerThreadPoolHolder.ServletContainerType.TOMCAT){
            //由于tomcat线程池在coreSize被占满的时候，会优先开辟新线程，而不是先进队列,故哲理poolSize取maxSize更合适
            poolSize = es.getMaximumPoolSize();//设置的最大线程数
        }

        poolSize = Math.max(coreSize, poolSize);
        if(perfTestRejectProperties.getThreadThreshold() > 0
            && activeCount * 1.0 / poolSize >= perfTestRejectProperties.getThreadThreshold()){ //线程池中的繁忙线程数过于饱和
            addRejectCount();
            log.warn("perfPoolServlet,coreSize:{},poolSize:{},activeCount:{}", coreSize, poolSize,activeCount);
            return true;
        }

        int queueSize = es.getQueue().size();
        if(perfTestRejectProperties.getQueueSizeThreshold() >= 0
                && queueSize > perfTestRejectProperties.getQueueSizeThreshold()){//任务开始堆积，直接拒绝压测请求
            addRejectCount();
            log.warn("perfPoolServlet,queueSize:{}", queueSize);
            return true;
        }

        return false;
    }

    /**
     * 判断压测请求是否熔断, 如果5秒内压测请求被拒绝的数量超过10个，则熔断5秒
     * @return
     */
    private boolean isCircuitBreak(){
        RejectedCountDto rejectedCountDto = servicePerfTestRejectedRequestCountCache.get("");

        if(rejectedCountDto.isCircuitBreak(perfTestRejectProperties.getCircuitBreakThreshold())){
            if(!rejectedCountDto.isLogged){
                rejectedCountDto.isLogged = true;

                //这个日志5秒最多打印一次，不会打印太多，以防影响性能
                log.warn("压测时发现servlet容器线程池过于繁忙，已熔断5秒内的所有压测请求");
            }
            return true;
        }

        return false;
    }

    /**
     * 添加压测请求拒绝数
     */
    private void addRejectCount(){
        RejectedCountDto rejectedCountDto = servicePerfTestRejectedRequestCountCache.get("");
        rejectedCountDto.incrRejectedCount();
    }

    @Override
    public void destroy() {
        //do nothing
    }
}
