package cn.com.duibaboot.ext.autoconfigure.web.container;

import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.MonitorRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.context.embedded.ConfigurableEmbeddedServletContainer;
import org.springframework.boot.context.embedded.EmbeddedServletContainerCustomizer;

import java.util.concurrent.*;

public abstract class SpecifiedEmbeddedServletContainerCustomizer<T extends ConfigurableEmbeddedServletContainer> implements EmbeddedServletContainerCustomizer, DisposableBean {
    protected final Logger logger = LoggerFactory.getLogger(getClass());
    public static final String HTTP_FAIL_FAST_THREAD_NAME_PREFIX = "HttpFailFastThread";

    private Class<T> factoryClass;

    //http容器的线程池
    private volatile ThreadPoolExecutor httpExecutor;
    private volatile ThreadPoolExecutor httpFailFastExecutor;
    private volatile ScheduledExecutorService scheduledExecutorService;


    public SpecifiedEmbeddedServletContainerCustomizer(Class<T> factoryClass){
        this.factoryClass = factoryClass;
    }

    @Override
    public void customize(ConfigurableEmbeddedServletContainer container) {
        if(factoryClass.isInstance(container)) {
            this.customizeSpecified((T)container);
        }
    }

    public abstract void customizeSpecified(T container);

    public abstract void shutdownGracefully();

    /**
     * 设置http线程池，设置后会开启一个线程扫描该线程池的队列，如果有http请求在队列中等待过久，则拒绝该http请求。
     * @param httpExecutor
     */
    public void failFastToHttpExecutor(ThreadPoolExecutor httpExecutor){
        this.httpExecutor = httpExecutor;
        startFailFastService();
    }

    @Override
    public void destroy() throws Exception {
        this.shutdownGracefully();
        stopFailFastService();

    }

    private BlockingQueue<Runnable> getQueueOfHttpExecutor(){
        ExecutorService temp = httpExecutor;
        ThreadPoolExecutor ex = ((ThreadPoolExecutor)temp);
        //we have idle threads, so return null
        if(ex instanceof TomcatCustomizer.TomcatMonitorThreadPoolExecutor) {
            if (((TomcatCustomizer.TomcatMonitorThreadPoolExecutor)ex).getSubmittedCount()
                    < (ex.getMaximumPoolSize())) {
                return null;
            }
        }else{
            if (ex.getActiveCount() < (ex.getMaximumPoolSize())) {
                return null;
            }
        }
        return ex.getQueue();
    }

    private void startFailFastService(){
        if(httpFailFastExecutor != null){
            //已经start过了，直接返回
            return;
        }
        httpFailFastExecutor = new ThreadPoolExecutor(1, 100, 60, TimeUnit.SECONDS, new ArrayBlockingQueue<>(100),
                new NamedThreadFactory(HTTP_FAIL_FAST_THREAD_NAME_PREFIX));
        scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(new NamedThreadFactory(
                "HttpFailFastScheduledThread"));

        //定时调度，判断是否有任务在http线程池工作队列中等待过久，如果等待超过指定毫秒数，则放入另一个线程池中执行
        //这个线程池中的线程都以HttpFailFastThread为前缀，然后在FailFastFilter中判断，如果线程前缀为HttpFailFastThread，则抛出RejectedExecutionException.
        //Http Rest RPC客户端判断到是此异常后，可以向其他服务器发起重试（因为此http调用并没有真正被处理，所以可以安全地发起调用）
        scheduledExecutorService.scheduleAtFixedRate(() -> {
            scanHttpThreadQueue();
        }, 1000, 10, TimeUnit.MILLISECONDS);
    }

    private void scanHttpThreadQueue(){//NOSONAR
        BlockingQueue<Runnable> workQueue = getQueueOfHttpExecutor();
        if(workQueue == null || workQueue.isEmpty()){
            return;
        }
        while(!workQueue.isEmpty()){
            //有http请求积压在请求队列里,判断如果等待时间超过50ms则直接拒绝。
            Runnable runnable = workQueue.peek();
            if(runnable == null){
                break;
            }

            boolean needMoveIntoAnotherPool = false;
            if(runnable instanceof MonitorRunnable) {
                MonitorRunnable mr = (MonitorRunnable) runnable;
                long createTimeMillis = mr.getSubmitTimeMillis();
                if(System.currentTimeMillis() - createTimeMillis > 50){//如果在队列中等待了50ms没有被处理，则放入另一个线程池中并尽快返回。
                    needMoveIntoAnotherPool = true;
                }
            }else{
                logger.warn("[NOTIFYME]runnable from httpExecutor is not instanceof MonitorRunnable.");// will not happen
            }

            if(needMoveIntoAnotherPool) {
                boolean removed = workQueue.remove(runnable);
                if (removed) {
                    try {
                        httpFailFastExecutor.execute(runnable);
                    } catch (RejectedExecutionException e) {
                        logger.warn("[NOTIFYME]put http thread into HttpFailFastThread failed", e);
                    }
                }
            }else{
                break;//第一个都没有超时，那么后面的就不用看了。
            }
        }
    }

    private void stopFailFastService(){
        if(scheduledExecutorService != null){
            scheduledExecutorService.shutdownNow();
        }
        if(httpFailFastExecutor != null){
            httpFailFastExecutor.shutdownNow();
        }
    }
}
