package cn.com.duibaboot.ext.autoconfigure.web.container;

import cn.com.duibaboot.ext.autoconfigure.core.DuibaServerProperties;
import cn.com.duibaboot.ext.autoconfigure.threadpool.proxy.MonitorRunnable;
import io.undertow.Undertow;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.springframework.boot.context.embedded.undertow.UndertowBuilderCustomizer;
import org.springframework.boot.context.embedded.undertow.UndertowEmbeddedServletContainerFactory;
import org.xnio.OptionMap;
import org.xnio.Options;
import org.xnio.Xnio;
import org.xnio.XnioWorker;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.concurrent.*;

/**
 * 自动配置Undertow等容器,增加http线程池监控，如果在队列中等待超过1秒，则打印error日志提示线程池过小<br/>
 * 如果是内部服务，则拒绝在http线程池工作队列中等待过久的任务，让客户端可以尽快重试<br/>
 *
 * 已经测试过，keepalive状态可以维持10W次连续请求，不像tomcat需要设置一个连接最多能接受的请求数。
 */
public class UndertowCustomizer extends SpecifiedEmbeddedServletContainerCustomizer<UndertowEmbeddedServletContainerFactory> {
    private static volatile DuibaServerProperties duibaServerProperties;
    private volatile XnioWorker xnioWorker;

    public UndertowCustomizer(DuibaServerProperties duibaServerProperties){
        super(UndertowEmbeddedServletContainerFactory.class);
        UndertowCustomizer.duibaServerProperties = duibaServerProperties;//NOSONAR
    }

    @Override
    public void customizeSpecified(UndertowEmbeddedServletContainerFactory container) {
        container.addBuilderCustomizers(new UndertowBuilderCustomizer() {
            @Override
            public void customize(Undertow.Builder builder) {
                try {
                    int ioThreads = (int) FieldUtils.readDeclaredField(builder, "ioThreads", true);
                    int workerThreads = (int)FieldUtils.readDeclaredField(builder, "workerThreads", true);
                    OptionMap.Builder workerOptions = (OptionMap.Builder)FieldUtils.readDeclaredField(builder, "workerOptions", true);
                    Xnio xnio = Xnio.getInstance(Undertow.class.getClassLoader());
                    XnioWorker worker = xnio.createWorker(OptionMap.builder()
                            .set(Options.WORKER_IO_THREADS, ioThreads)
                            .set(Options.CONNECTION_HIGH_WATER, 1000000)
                            .set(Options.CONNECTION_LOW_WATER, 1000000)
                            .set(Options.WORKER_TASK_CORE_THREADS, workerThreads)
                            .set(Options.WORKER_TASK_MAX_THREADS, workerThreads)
                            .set(Options.TCP_NODELAY, true)
                            .set(Options.CORK, true)
                            .addAll(workerOptions.getMap())
                            .getMap());

                    //以下代码本来是希望拦截线程池的execute方法来把Runnable封装成MonitorRunnable，无奈xnio.createWorker内部会直接引用已经构造好的线程池，所以这样设置无效，只能用javaagent修改字节码来达到类似目的
//                    ProxyFactory proxyFactory = new ProxyFactory();
//                    proxyFactory.setTargetSource(new SingletonTargetSource(worker){
//                        @Override
//                        public Class<?> getTargetClass() {
//                            return XnioWorker.class;
//                        }
//                    });
//                    proxyFactory.addAdvice(new XnioWorkerMethodInterceptor());
//                    XnioWorker proxyedWorker = (XnioWorker)proxyFactory.getProxy();
//
                    builder.setWorker(worker);
                    xnioWorker = worker;

                    //如果因为undertow api改变了，导致这里拿taskPool或转型失败，则需要对应地修改UndertowInstrumentation类。
                    Field taskPoolField = FieldUtils.getDeclaredField(XnioWorker.class, "taskPool", true);
                    ThreadPoolExecutor httpExecutor = (ThreadPoolExecutor) taskPoolField.get(worker);

                    EmbeddedServletContainerThreadPoolHolder.setServletContainerThreadPool(httpExecutor);
                    EmbeddedServletContainerThreadPoolHolder.setServletContainerType(EmbeddedServletContainerThreadPoolHolder.ServletContainerType.UNDERTOW);

                    //如果是内部服务，则开启异步扫描线程，扫描出所有在队列中等待过久的任务，直接拒绝，以达到failfast的目的.（配合FailFastFilter共同生效）
                    if(duibaServerProperties.isInternalMode()){
                        failFastToHttpExecutor(httpExecutor);
                    }
                } catch (IllegalAccessException | IOException e) {
                    throw new RuntimeException("[NOTIFYME] will never be here", e);
                }
            }
        });
    }

    @Override
    public void shutdownGracefully(){
        if(xnioWorker != null) {
            xnioWorker.shutdown();
            try {
                xnioWorker.awaitTermination(5000, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                //Ignore
                Thread.currentThread().interrupt();
            }
            if(!xnioWorker.isTerminated()) {
                xnioWorker.shutdownNow();
            }

            xnioWorker = null;
        }
    }

    private static class XnioWorkerMethodInterceptor implements MethodInterceptor {

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            Method m = invocation.getMethod();
            if(m.getName().equals("execute")) {//提交线程到线程池的方法

                Runnable runnable = (Runnable)invocation.getArguments()[0];
                runnable = new MonitorRunnable(runnable);
                invocation.getArguments()[0] = runnable;

                return invocation.proceed();
            }

            return invocation.proceed();
        }
    }

}
