package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;
import java.util.*;

/**
 * 引流回归一些工具方法
 * Created by guoyanfei .
 * 2019-03-14 .
 */
public final class FlowReplayUtils {

    private static final String DUIBA_FLOW_REPLAY_REPORT_ID = System.getenv("DUIBA_FLOW_REPLAY_REPORT_ID");

    private FlowReplayUtils() {
    }

    /**
     * 判断一个字符串是否是json
     * @param json
     * @return
     */
    public static boolean isJSONValid(String json) {
        if (json == null) {
            return false;
        }
        if (!json.startsWith("{") && !json.startsWith("[")) {
            return false;
        }
        try {
            JSONObject.parseObject(json);
        } catch (Exception e0) {
            try {
                JSONObject.parseArray(json);
            } catch (Exception e1) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断是否是回放专用环境
     * @return
     */
    public static boolean isReplayEnv() {
        return StringUtils.isNotBlank(DUIBA_FLOW_REPLAY_REPORT_ID);
    }

    /**
     * 判断报告id是否合法
     * @param reportId
     * @return
     */
    public static boolean isReportIdValid(Long reportId) {
        if (StringUtils.isBlank(DUIBA_FLOW_REPLAY_REPORT_ID)) {
            return false;
        }
        return DUIBA_FLOW_REPLAY_REPORT_ID.equals(String.valueOf(reportId));
    }

    /**
     * 根据方法，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param method
     * @return
     */
    public static String parseApiNameByMethod(Method method) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, method.getDeclaringClass());
        sb.append(".");
        sb.append(method.getName());
        sb.append("(");
        Class<?>[] parametersTypes = method.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    /**
     * 调用的方法是否一样
     * @param spanMethod
     * @param currentMethod
     * @return
     */
    public static boolean isMethodEqual(String spanMethod, Method currentMethod) {
        return spanMethod.equals(currentMethod.getName());
    }

    /**
     * 调用的参数类型是否一样
     * @param spanParameterTypes
     * @param currentParameterTypes
     * @return
     */
    public static boolean isArgumentsTypesEqual(String[] spanParameterTypes, Class<?>[] currentParameterTypes) {
        for (int i = 0; i < currentParameterTypes.length; i++) {
            if (!currentParameterTypes[i].getName().equals(spanParameterTypes[i])) {
                return false;
            }
        }
        return true;
    }

    public static String stringArrayToString(String[] stringArray) {
        if (stringArray == null || stringArray.length == 0) {
            return StringUtils.EMPTY;
        }
        return Joiner.on(",").join(stringArray);
    }

    public static String classArrayToString(Class<?>[] classArray) {
        if (classArray == null || classArray.length == 0) {
            return StringUtils.EMPTY;
        }
        List<String> classNames = new ArrayList<>(classArray.length);
        for (Class<?> c : classArray) {
            classNames.add(c.getName());
        }
        return Joiner.on(",").join(classNames);
    }

    /**
     * 调用的参数值是否一样
     * @param spanParameterValues
     * @param currentParameterValues
     * @return
     */
    public static boolean isArgumentsEqual(Object[] spanParameterValues, Object[] currentParameterValues) {
        for (int i = 0; i < currentParameterValues.length; i++) {
            Object arg = currentParameterValues[i];
            Object spanArg = spanParameterValues[i];
            if (!FlowReplayUtils.isObjectEqual(JSON.toJSON(spanArg), JSON.toJSON(arg))) {
                return false;
            }
        }
        return true;
    }

    /**
     * 方法的返回值类型是否一样
     * @param spanReturnType
     * @param currentReturnType
     * @return
     */
    public static boolean isReturnTypeEqual(String spanReturnType, Class<?> currentReturnType) {
        if (spanReturnType == null && currentReturnType == null) {
            return true;
        }
        if (currentReturnType != null) {
            return currentReturnType.getName().equals(spanReturnType);
        }
        return false;
    }

    /**
     * 方法所在类全路径是不是一样
     * @param spanTypeFullPath
     * @param currentTypeFullPath
     * @return
     */
    public static boolean isTypeFullPathEqual(String spanTypeFullPath, String currentTypeFullPath) {
        return Objects.equals(spanTypeFullPath, currentTypeFullPath);
    }

    /**
     * 全局拦截白名单，白名单的不拦截
     * @param allArguments
     * @return
     */
    public static boolean isGlobalWhitelist(Object[] allArguments) {
        if (allArguments != null) {
            for (Object obj : allArguments) {
                if (obj == null) {
                    continue;
                }
                if (FlowReplayConstants.CANNOT_DESERIALIZE_CLASSES.contains(obj.getClass().getName())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 判断两个对象是否一样
     * @param recordVal
     * @param replayVal
     * @return
     */
    public static boolean isObjectEqual(Object recordVal, Object replayVal) {
        if (recordVal == null && replayVal == null) {
            return true;
        }
        if (recordVal instanceof byte[] && replayVal instanceof byte[]) {
            return Arrays.equals((byte[]) recordVal, (byte[]) replayVal);
        }
        if (recordVal instanceof String && replayVal instanceof String) {
            return recordVal.equals(replayVal);
        }
        if (recordVal instanceof JSONArray && replayVal instanceof JSONArray) {
            return isJSONArrayEqual((JSONArray) recordVal, (JSONArray) replayVal);
        }
        if (recordVal instanceof JSONObject && replayVal instanceof JSONObject) {
            return isJSONObjectEqual((JSONObject) recordVal, (JSONObject) replayVal);
        }

        if (recordVal != null) {
            return recordVal.equals(replayVal);
        }
        return false;
    }

    private static boolean isJSONArrayEqual(JSONArray recordArray, JSONArray replayArray) {
        if (recordArray.size() != replayArray.size()) {
            return false;
        }

        // 遍历录制返回值数组，每一项和回放返回值数组对应的项进行比对
        Iterator<Object> iterator = recordArray.iterator();
        for (int i = 0; iterator.hasNext(); i++) {
            Object recordValue = iterator.next();
            String recordValueStr = String.valueOf(recordValue);

            if (FlowReplayUtils.isJSONValid(recordValueStr)) { // 如果是jsonObj或者jsonArray，那么再判断内部的东西
                return isObjectEqual(JSON.parse(recordValueStr), JSON.parse(String.valueOf(recordArray.get(i))));
            } else {    // 如果不是jsonObj或者jsonArray，那么直接进行值对比，如果值不一样，添加对比不一致的结果
                if (!Objects.equals(recordValue, replayArray.get(i))) {
                    return false;
                }
            }
        }
        return true;
    }

    private static boolean isJSONObjectEqual(JSONObject recordObj, JSONObject replayObj) {
        // 遍历录制返回值对象，每一个key的value和回放返回值对象对应的value进行比对
        for (Map.Entry<String, Object> entry : recordObj.entrySet()) {
            Object recordValue = entry.getValue();
            Object replayValue = replayObj.get(entry.getKey());
            String recordValueStr = String.valueOf(recordValue);
            String replayValueStr = String.valueOf(replayValue);

            if (FlowReplayUtils.isJSONValid(recordValueStr)) {  // 当前遍历到的value，是个jsonObj或者jsonArray，那么需要递归判断里面的内容
                return isObjectEqual(JSON.parse(recordValueStr), JSON.parse(replayValueStr));
            } else {    // 当前遍历到的value，不是jsonObj或者jsonArray，那么直接进行值的比对，如果值不一样，添加对比不一致的结果
                if (!Objects.equals(recordValue, replayValue)) {
                    return false;
                }
            }
        }
        return true;
    }


    /**
     * 根据方法签名，解出类似以下格式的字符串
     * RemoteConsumerService.getConsumer(Long)
     * @param methodSignature
     * @return
     */
    public static String parseApiNameByMethodSignature(MethodSignature methodSignature) {
        StringBuilder sb = new StringBuilder();
        appendType(sb, methodSignature.getDeclaringType());
        sb.append(".");
        sb.append(methodSignature.getMethod().getName());
        sb.append("(");
        Class<?>[] parametersTypes = methodSignature.getParameterTypes();
        appendTypes(sb, parametersTypes);
        sb.append(")");
        return sb.toString();
    }

    private static void appendTypes(StringBuilder sb, Class<?>[] types) {
        for (int size = types.length, i = 0; i < size; i++) {
            appendType(sb, types[i]);
            if (i < size - 1) {
                sb.append(",");
            }
        }
    }

    private static void appendType(StringBuilder sb, Class<?> type) {
        if (type.isArray()) {
            appendType(sb, type.getComponentType());
            sb.append("[]");
        } else {
            sb.append(type.getSimpleName());
        }
    }
}
