package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordResultBuilder;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordTraceWriter;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.RecordCustomizeFlowReplaySpanAspect;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.RecordExcludeHandlerInterceptor;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.RecordRemoteServicePlugin;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.RecordSpringMvcFilter;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordResultMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordStopMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler.PercentageBasedRecordSampler;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler.RecordSampler;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop.ReplayCustomizeFlowReplaySpanAspect;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop.ReplaySpringMvcFilter;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayResultMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayStopMvcEndpoint;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.ConcreateReplayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.RemoteServiceReplayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.Replayer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.SpringMvcReplayer;
import cn.com.duibaboot.ext.autoconfigure.security.BaseSecurityManager;
import cn.com.duibaboot.ext.autoconfigure.security.SecurityManagerAutoConfiguration;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.HttpClient;

import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.cloud.client.loadbalancer.RestTemplateCustomizer;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.*;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.ByteArrayHttpMessageConverter;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.security.Permission;
import java.util.EnumSet;
import java.util.List;

/**
 * 引流回归自动配置
 * Created by guoyanfei .
 * 2019-01-23 .
 */
@Lazy
@Configuration
public class FlowReplayAutoConfiguration {

    /**
     * 录制配置
     */
    @Configuration
    static class RecordConfiguration {

        @Bean
        public RecordCustomizeFlowReplaySpanAspect recordCustomizeFlowReplaySpanAspect(){
            return new RecordCustomizeFlowReplaySpanAspect();
        }

        /**
         * /flow/record 端点
         * 开始录制
         *
         * @return
         */
        @Bean
        public RecordMvcEndpoint recordMvcEndpoint() {
            return new RecordMvcEndpoint();
        }

        /**
         * /flow/record/stop 端点
         * 结束录制
         *
         * @return
         */
        @Bean
        public RecordStopMvcEndpoint recordStopMvcEndpoint() {
            return new RecordStopMvcEndpoint();
        }

        /**
         * /flow/record/result 端点
         * 查询录制结果
         *
         * @return
         */
        @Bean
        public RecordResultMvcEndpoint recordResultMvcEndpoint() {
            return new RecordResultMvcEndpoint();
        }

        /**
         * 录制结果构建器
         * 录制的用例集文件压缩上传，并且构建结果回填给 RecordContext
         *
         * @return
         */
        @Bean
        public RecordResultBuilder recordResultBuilder() {
            return new RecordResultBuilder();
        }

        /**
         * SpringMvc 控制器方法的过滤器
         * 录制http的请求
         *
         * @return
         */
        @Bean
        public RecordSpringMvcFilter recordSpringMvcFilter() {
            return new RecordSpringMvcFilter();
        }

        /**
         * 录制结果写到文件中
         * 把内存队列中的录制结果写到文件中
         * @return
         */
        @Bean
        public RecordTraceWriter recordTraceWriter() {
            return new RecordTraceWriter();
        }

        @Bean
        public FilterRegistrationBean recordSpringMvcFilterConfigurer(RecordSpringMvcFilter recordSpringMvcFilter){
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(recordSpringMvcFilter);
            List<String> urlPatterns= Lists.newArrayList();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(0);
            return registrationBean;
        }

        @Configuration
        @ConditionalOnClass({ Servlet.class, RecordExcludeHandlerInterceptor.class, InterceptorRegistry.class})
        @ConditionalOnWebApplication
        @Order(-100)
        public static class RecordExcludeHandlerInterceptorConfiguration extends WebMvcConfigurerAdapter {

            @Override
            public void addInterceptors(InterceptorRegistry registry) {
                RecordExcludeHandlerInterceptor interceptor = new RecordExcludeHandlerInterceptor();
                registry.addInterceptor(interceptor);
                super.addInterceptors(registry);
            }

        }
        @Configuration
        static class FlowRecordAspectConfiguration {

            /**
             * remoteService录制拦截器配置
             *
             * @return
             */
            @Bean
            public RecordRemoteServicePlugin recordRemoteServicePlugin() {
                return new RecordRemoteServicePlugin();
            }

        }

        /**
         * 录制采样器配置
         */
        @Bean
        @ConditionalOnMissingBean
        public RecordSampler recordSampler() {
            return new PercentageBasedRecordSampler();
        }
    }

    /**
     * 回放配置
     */
    @Configuration
    @Conditional(ReplayCondition.class)
    static class ReplayConfiguration {

        @Bean
        public ReplayCustomizeFlowReplaySpanAspect replayCustomizeFlowReplaySpanAspect(){
            return new ReplayCustomizeFlowReplaySpanAspect();
        }

        /**
         * /flow/repaly 端点
         * 开始回放
         *
         * @return
         */
        @Bean
        public ReplayMvcEndpoint replayMvcEndpoint() {
            return new ReplayMvcEndpoint();
        }

        /**
         * /flow/replay/stop 端点
         * 终止回放
         *
         * @return
         */
        @Bean
        public ReplayStopMvcEndpoint replayStopMvcEndpoint() {
            return new ReplayStopMvcEndpoint();
        }

        /**
         * /flow/replay/result 端点
         * 查询回放结果
         * @return
         */
        @Bean
        public ReplayResultMvcEndpoint replayResultMvcEndpoint() {
            return new ReplayResultMvcEndpoint();
        }

        /**
         * 用例集加载器的配置
         * 下载并解压录制的用例集文件，然后把用例读取到内存中，供用例消费者消费
         * @return
         */
        @Bean
        public ReplayTraceLoader replayTraceLoader() {
            return new ReplayTraceLoader();
        }

        /**
         * 用例消费者配置
         * 多线程消费用例加载器加载到内存中的用例，调用回放器回放对应类型的用例
         * @return
         */
        @Bean
        public ReplayTraceReplayer replayTraceReplayer() {
            return new ReplayTraceReplayer();
        }

        /**
         * 回放结果写到文件中
         * 把内存队列中的回放结果写到文件中
         * @return
         */
        @Bean
        public ReplayTraceResultWriter replayTraceResultWriter() {
            return new ReplayTraceResultWriter();
        }

        /**
         * 回放结果构建器
         * 回放的结果文件压缩上传，并且构建结果回填给 ReplayContext
         *
         * @return
         */
        @Bean
        public ReplayResultBuilder replayResultBuilder() {
            return new ReplayResultBuilder();
        }

        /**
         * 回放器配置
         * 目前支持 RemoteService 和 SpringMvc Controller两种调用的回放
         */
        @Configuration
        static class ReplayerConfiguration {

            @Bean
            public RemoteServiceReplayer remoteServiceReplayer() {
                return new RemoteServiceReplayer();
            }

            @Bean
            public SpringMvcReplayer springMvcReplayer() {
                return new SpringMvcReplayer();
            }

            @Bean
            public Replayer replayer() {
                return new ConcreateReplayer();
            }

        }

        /**
         * SpringMvc 控制器方法的过滤器
         * 回放http的请求设置上下文
         *
         * @return
         */
        @Bean
        public ReplaySpringMvcFilter replaySpringMvcFilter() {
            return new ReplaySpringMvcFilter();
        }

        @Bean
        public FilterRegistrationBean replaySpringMvcFilterConfigurer(ReplaySpringMvcFilter replaySpringMvcFilter){
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(replaySpringMvcFilter);
            List<String> urlPatterns= Lists.newArrayList();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(0);
            return registrationBean;
        }

        /**
         * 设置SecurityManager，以支持禁止访问部分网络的功能
         * @return
         */
        @Bean
        public DuibaSecurityManagerConfiguar4ReplayApplicationListener duibaSecurityManagerConfiguar4ReplayApplicationListener(){
            return new DuibaSecurityManagerConfiguar4ReplayApplicationListener();
        }

        @Slf4j
        static class DuibaSecurityManagerConfiguar4ReplayApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

            private boolean flag = true;

            @Override
            public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
                if(!flag) {
                    return;
                }
                try{
                    System.setSecurityManager(new BaseSecurityManager(){
                        @Override
                        public void checkExec(String cmd) {
                            //允许执行任意shell
                        }

                        @Override
                        public void checkPermission(Permission perm) {
                            //允许任意Permission, 除了不允许调用setSecurityManager
                            if("setSecurityManager".equals(perm.getName())){
                                super.checkPermission(perm);
                            }
                        }

                        @Override
                        public void checkConnect(String host, int port) {//判断是否可以连接远程host
                            if (ReplayContextHolder.canHostPass(host)) {
                                return;
                            }
                            throw new SecurityException("Not allowed to connect to host:" + host + ",port:" + port);
                        }

                        @Override
                        public void checkConnect(String host, int port, Object context) {
                            this.checkConnect(host, port);

                        }
                    });
                } catch(Throwable e){
                    log.error("设置SercurityManager失败，这会导致回放时仍然能访问网络，可能会污染数据库数据，退出当前应用", e);
                    System.exit(-1);
                }

                flag = false;
            }

            @Override
            public int getOrder() {
                return SecurityManagerAutoConfiguration.ORDER_OF_CONTEXT_REFRESHED_EVENT - 1;
            }
        }
    }

    private static class ReplayCondition implements Condition {

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            return FlowReplayUtils.isReplayEnv();
        }

    }

    /**
     * 文件操作组件
     * 使用 loadBalancedRestTemplate
     * @param httpClient
     * @param customizers
     * @return
     */
    @Bean
    public FlowReplayFileComponent flowReplayFileComponent(HttpClient httpClient, List<RestTemplateCustomizer> customizers) {
        RestTemplate loadBalancedRestTemplate = loadBalancedRestTemplate(httpClient);
        for (RestTemplateCustomizer customizer : customizers) {
            customizer.customize(loadBalancedRestTemplate);
        }
        return new FlowReplayFileComponent(loadBalancedRestTemplate);
    }

    private RestTemplate loadBalancedRestTemplate(HttpClient httpClient) {
        HttpComponentsClientHttpRequestFactory httpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        httpRequestFactory.setConnectionRequestTimeout(3000);
        httpRequestFactory.setConnectTimeout(3000);
        httpRequestFactory.setReadTimeout(60000);
        httpRequestFactory.setHttpClient(httpClient);
        RestTemplate template = new RestTemplate(httpRequestFactory);
        template.getMessageConverters().add(new ByteArrayHttpMessageConverter());
        return template;
    }

}
