package cn.com.duibaboot.ext.autoconfigure.flowreplay.record;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordConfigDto;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer.Hessian2Serializer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * 实例的录制上下文
 * Created by guoyanfei .
 * 2019-01-23 .
 */
@Slf4j
public final class RecordContext {

    private static final int TRACE_QUEUE_SIZE = 500;

    /**
     * 用例集id
     */
    private final Long usecaseSetId;

    /**
     * 录制文件最大分钟数，单位分钟
     */
    private final Integer maxMinutes;

    private volatile Long startTime;

    private volatile Long endTime;

    /**
     * 录制文件最大大小，单位M
     */
    private final Integer maxFileSize;

    /**
     * 录制文件当前大小，单位byte
     */
    private long currentFileSize;

    private RandomAccessFile raf;

    /**
     * trace的队列，每个请求录制完成往traceQueue中offer一条，另外有一个线程逐个poll traceQueue中的内容写文件
     */
    private LinkedBlockingQueue<FlowReplayTrace> traceQueue = new LinkedBlockingQueue<>(TRACE_QUEUE_SIZE);

    /**
     * 录制结束(录制文件还没写入完成)标记
     */
    private volatile boolean recordFinished = false;

    /**
     * 录制结果
     */
    private volatile RecordResult recordResult;

    public RecordContext(RecordConfigDto recordConfig) {
        if (recordConfig == null || !recordConfig.isLegal()) {
            throw new FlowReplayException("录制配置为空or录制参数不合法");
        }
        this.usecaseSetId = recordConfig.getUsecaseSetId();

        this.maxMinutes = recordConfig.getMaxMinutes();
        this.startTime = null;
        this.endTime = null;

        this.maxFileSize = recordConfig.getMaxFileSize();
        this.currentFileSize = 0L;

        this.recordResult = null;

    }

    /**
     * 判断时间是否已经录制到需要结束的时候
     * @return
     */
    public boolean isTimeToEnd() {
        if (this.startTime == null || this.maxMinutes == null) {
            log.error("录制开始时间为空or最大录制分钟数为空_返回需要结束录制");
            return true;
        }
        long expectEndTime = startTime + maxMinutes * 60 * 1000;
        return expectEndTime < System.currentTimeMillis();
    }

    /**
     * 判断文件大小是否已经录制到需要结束的时候
     * @return
     */
    public boolean isFileSizeToEnd() {
        if (this.maxFileSize == null) {
            log.error("录制文件的最大限制为空_返回需要结束录制");
            return true;
        }
        int expectFileSize = this.maxFileSize * 1024 * 1024;
        return expectFileSize < this.currentFileSize;
    }

    /**
     * 初始化录制文件
     */
    private synchronized void initRecordFile() {
        File file = new File(FlowReplayConstants.LOCAL_RECORD_FILEPATH);
        if (file.exists()) {
            return;
        }
        try {
            raf = new RandomAccessFile(file, "rw");
            raf.writeInt(FlowReplayConstants.RECORD_FILE_VERSION);  // 录制文件的版本号
            raf.writeInt(0);    // 文件初始化，用例数设置为0
        } catch (IOException e) {
            log.error("初始化RandomAccessFile异常", e);
            throw new FlowReplayException("初始化RandomAccessFile异常");
        }
    }

    private synchronized void closeFile() {
        if (raf != null) {
            IOUtils.closeQuietly(raf);
        }
        raf = null;
    }

    /**
     * 写trace到本地文件
     * @param trace
     * @throws IOException
     */
    public synchronized void writeTrace(FlowReplayTrace trace) throws IOException {
        if (raf == null) {
            return;
        }
        if (trace == null) {
            return;
        }
        byte[] traceBytes = Hessian2Serializer.serialize(trace);
        raf.writeInt(traceBytes.length);    // 写入单个trace的长度，读取的时候先读取长度，然后根据长度，放心读取trace
        raf.write(traceBytes);              // 写入trace
        this.currentFileSize = raf.getFilePointer(); // 设置录制文件的当前长度到内存
    }

    /**
     * 开始录制
     */
    public void start() {
        initRecordFile();     // 初始化本次录制的文件

        this.startTime = System.currentTimeMillis();
    }

    /**
     * 结束录制
     */
    public void end() {
        this.closeFile();

        endTime = System.currentTimeMillis();
    }

    public void setRecordResult(RecordResult recordResult) {
        this.recordResult = recordResult;
    }

    public Long getUsecaseSetId() {
        return usecaseSetId;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public long getCurrentFileSize() {
        return currentFileSize;
    }

    public RecordResult getRecordResult() {
        return recordResult;
    }

    public void offerTrace(FlowReplayTrace trace) {
        this.traceQueue.offer(trace);
    }

    public FlowReplayTrace pollTrace() throws InterruptedException {
        return this.traceQueue.poll(2, TimeUnit.SECONDS);
    }

    public boolean isTraceQueueEmpty() {
        return this.traceQueue.size() < 1;
    }

    public void recordFinished() {
        this.recordFinished = true;
    }

    public boolean isRecordFinished() {
        return this.recordFinished;
    }
}
