package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.RedisFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 流量录制的redis方法拦截器
 * 把方法名称，参数列表，参数值列表，都录制下来
 */
@Slf4j
public class RecordJedisMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以录制
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canRecord(Method method, Object[] allArguments) {
        if (!FlowReplayTrace.isTraced()) {
            return false;
        }
        if (FlowReplayConstants.REDIS_AOP_EXCLUDE_METHOD.contains(method.getName())) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) {
        if (!canRecord(method, allArguments)) {
            return;
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return;
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method, allArguments);
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) {
        if (!canRecord(method, allArguments)) {
            return ret;
        }
        if (!IgnoreSubInvokesContext.isInstMarked(obj, method, allArguments)) {
            return ret;
        }
        try {
            RedisFlowReplaySpan span = RedisFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, ret);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
            log.debug("Redis录制_traceId={}_spanId={}", span.getTraceId(), span.getSpanId());
        } catch (Throwable t) {
            log.error("Redis_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }

    @Override
    public void handleMethodException(Object objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // 如果捕捉到异常，并且正在录制中，那么不录这个用例
        if (FlowReplayTrace.isTraced()) {
            FlowReplayTrace.remove();
        }
    }
}
