package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler;

import java.util.BitSet;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 最小支持0.0001F（即万分之一的采样率）
 * This sampler is appropriate for low-traffic instrumentation (ex servers that each receive <100K
 * requests), or those who do not provision random trace ids. It not appropriate for collectors as
 * the sampling decision isn't idempotent (consistent based on trace id).
 *
 * <h3>Implementation</h3>
 *
 * <p>Taken from <a href="https://github.com/openzipkin/zipkin-java/blob/traceid-sampler/zipkin/src/main/java/zipkin/CountingTraceIdSampler.java">Zipkin project</a></p>
 *
 * <p>This counts to see how many out of 10000 traces should be retained. This means that it is
 * accurate in units of 10000 traces.
 *
 * @author Marcin Grzejszczak
 * @author Adrian Cole
 * @since 1.0.0
 */
public class PercentageBasedRecordSampler implements RecordSampler {

    private final    AtomicInteger counter = new AtomicInteger(0);
    private volatile BitSet        sampleDecisions;
    private volatile float         percentage;

    public PercentageBasedRecordSampler() {
        init(0.01f);
    }

    public void init(Float percentage) {
        // 如果参数没传，用默认
        if (percentage == null) {
            return;
        }
        this.percentage = percentage;
        int outOf10000 = (int) (this.percentage * 10000.0f);
        this.sampleDecisions = randomBitSet(10000, outOf10000, new Random());
    }

    @Override
    public boolean isSampled() {
        if (this.percentage == 0) {  //NOSONAR
            return false;
        } else if (this.percentage == 1.0f) {    //NOSONAR
            return true;
        }
        synchronized (this) {
            final int i = this.counter.getAndIncrement();
            boolean result = this.sampleDecisions.get(i);
            if (i == 9999) {
                this.counter.set(0);
            }
            return result;
        }
    }

    /**
     * Reservoir sampling algorithm borrowed from Stack Overflow.
     * http://stackoverflow.com/questions/12817946/generate-a-random-bitset-with-n-1s
     */
    static BitSet randomBitSet(int size, int cardinality, Random rnd) {
        BitSet result = new BitSet(size);
        int[] chosen = new int[cardinality];
        int i;
        for (i = 0; i < cardinality; ++i) {
            chosen[i] = i;
            result.set(i);
        }
        for (; i < size; ++i) {
            int j = rnd.nextInt(i + 1);
            if (j < cardinality) {
                result.clear(chosen[j]);
                result.set(i);
                chosen[j] = i;
            }
        }
        return result;
    }

}
