package cn.com.duibaboot.ext.autoconfigure.security;

import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import sun.security.util.SecurityConstants;

import javax.annotation.PostConstruct;
import javax.security.auth.AuthPermission;
import java.io.FilePermission;
import java.io.SerializablePermission;
import java.lang.reflect.ReflectPermission;
import java.net.NetPermission;
import java.net.SocketPermission;
import java.security.AccessControlException;
import java.security.AllPermission;
import java.security.Permission;
import java.security.SecurityPermission;
import java.sql.SQLPermission;
import java.util.Arrays;
import java.util.HashSet;
import java.util.PropertyPermission;
import java.util.Set;
import java.util.logging.LoggingPermission;

/**
 * 自动配置SecurityManager，防止shell攻击（即使有fastjson等存在shell漏洞的框架也能有效防止）
 * http://www.cnblogs.com/yiwangzhibujian/p/6207212.html
 * Created by wenqi.huang on 2017/4/5.
 */
@Configuration
@ConditionalOnProperty(name = "duiba.securitymanager.enable", havingValue="true",  matchIfMissing = true)
@EnableConfigurationProperties(SecurityProperties.class)
public class SecurityManagerAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(SecurityManagerAutoConfiguration.class);

    public static final int ORDER_OF_CONTEXT_REFRESHED_EVENT = -10;

    static Set<String> forbiddenPermNames = new HashSet<>(10);

    @Autowired
    private SecurityProperties securityProperties;

    private Set<String> whiteShells = new HashSet<>();

    @PostConstruct
    public void init(){
        whiteShells.addAll(Arrays.asList(new String[]{
                "/usr/bin/id", "/bin/id", "id", "/usr/xpg4/bin/id", //netty需要执行的命令
                "setsid", //hbase需要执行的命令
                "jstat", "jmap"
        }));

        String[] forbiddenShells = new String[]{"sh","bash","source","exec","fork"};//必须禁止的shell
        String str = securityProperties.getWhiteShellList();
        String[] arr = StringUtils.split(str, ',');
        if(arr == null){
            return;
        }
        for(String shell : arr){
            if(isForbiddenShell(shell, forbiddenShells)){
                logger.warn("你设置的shell白名单(duiba.securitymanager.whiteShellList)中包含过于危险的shell：`{}`, 已排除.", shell);
            } else{
                whiteShells.add(shell);
            }
        }
    }

    private boolean isForbiddenShell(String shell, String[] forbiddenShells){
        for(String forbiddenShell : forbiddenShells) {
            if (forbiddenShell.equalsIgnoreCase(shell)){
                return true;
            }
        }
        return false;
    }

    static{
        //forbiddenPermNames.add("stopThread");
        forbiddenPermNames.add("queuePrintJob");
        //访问https url时jdk会创建 JCESecurityManager
        //forbiddenPermNames.add("createSecurityManager");

        //测试模式不屏蔽下面这些权限。否则运行单测会报错
        if(!SpringBootUtils.isUnitTestMode()) {//,"org.gradle.api.internal.tasks.testing.worker.TestWorker"
            forbiddenPermNames.add("setIO");
            forbiddenPermNames.add("setSecurityManager");
        }
    }

    //指定的类都不存在时返回true，有一个存在则返回false
    private static boolean conditionalOnMissingClass(String... classNames) {
        boolean existsOne = false;
        for(String className : classNames) {
            try {
                Class.forName(className);
                existsOne = true;
                break;
            } catch (ClassNotFoundException e) {
                //Ignore
            }
        }

        return !existsOne;
    }

    @Bean
    public DuibaSecurityManagerConfiguarApplicationListener duibaSecurityManagerConfiguarApplicationListener(){
        return new DuibaSecurityManagerConfiguarApplicationListener();
    }

    class DuibaSecurityManagerConfiguarApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

        private boolean flag = true;

        @Override
        public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
            if(!flag) {
                return;
            }
            if(System.getSecurityManager() == null) {
                System.setSecurityManager(new CustomSecurityManager());
            }

            flag = false;
        }

        @Override
        public int getOrder() {
            return ORDER_OF_CONTEXT_REFRESHED_EVENT;
        }
    }

    private class CustomSecurityManager extends BaseSecurityManager {
        private boolean isInWhiteShellList(String cmd){
            if(whiteShells.contains(cmd)){
                return true;
            }
            return false;
        }

        @Override
        public void checkExec(String cmd) {
            if(isInWhiteShellList(cmd)){
                return;
            }

            //不允许执行shell脚本
            try {
                super.checkExec(cmd);
            }catch(AccessControlException e){
                logger.warn("some one try to execute shell:`{}`, block it", cmd);
                throw e;
            }
        }

        @Override
        public void checkPermission(Permission perm) {
//                                Class<?>[] classes = this.getClassContext();
//                                StackTraceElement[] st = new RuntimeException().getStackTrace();
            //这里判断Permission类型的方法，如果换成Set.contains判断，速度会降低2000倍，所以用if来判断

            if(perm.getName().equals("getClassLoader")
                    || perm.getName().equals("getProtectionDomain")
                    || perm.getName().equals("accessDeclaredMembers")
                    || perm.getName().equals("<all permissions>")
                    || perm.getClass().getName().equals("java.lang.RuntimePermission")){//NOSONAR
                return;
            }

            //这里不能使用instanceof来判断，如果这么做了，会与bytebuddy冲突，导致抛出java.lang.ClassCircularityError
            if(perm.getClass().getName().equals("java.io.FilePermission")){//NOSONAR
                FilePermission fp = (FilePermission)perm;
                if(fp.getActions().equals(SecurityConstants.FILE_EXECUTE_ACTION)){
                    super.checkPermission(perm);
                    return;
                }
            }

            if(perm instanceof AllPermission
                    || perm instanceof PropertyPermission
                    || perm instanceof SerializablePermission
                    || perm instanceof ReflectPermission
                    || perm instanceof NetPermission
                    || perm instanceof SQLPermission
                    || perm instanceof SecurityPermission
                    || perm instanceof LoggingPermission
                    || perm instanceof AuthPermission
                    || perm instanceof SocketPermission){
                return;
            }

            if(forbiddenPermNames.contains(perm.getName())) {
                super.checkPermission(perm);
                return;
            }
        }

        @Override
        public void checkConnect(String host, int port) {
            //允许连接任意host
        }

        @Override
        public void checkConnect(String host, int port, Object context) {
            //允许连接任意host
        }

    }
}
