package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import java.util.List;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.perftest.DubboPerfTestRegistryFactoryWrapper;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ZoneAvoidanceRule;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FilterBasedRule extends ZoneAvoidanceRule {

	protected Logger log = LoggerFactory.getLogger(getClass());

	private List<RibbonServerListFilter> serverListFilters;

	public FilterBasedRule(List<RibbonServerListFilter> serverListFilters){
		this.serverListFilters = serverListFilters;
	}

	@Override
	public Server choose(Object key) {
		ILoadBalancer lb = getLoadBalancer();
		if (lb == null) {
			log.warn("no load balancer");
			return null;
		}
		//注意这里调用的是getReachableServers方法，只会拿到isAlive的服务器列表进行选择
		List<Server> servers = getPredicate().getEligibleServers(lb.getReachableServers(), key);

		if(serverListFilters != null) {
			for (RibbonServerListFilter filter : serverListFilters) {
				servers = filter.filter(servers, key);
			}
		}

		if(servers.isEmpty()){
			log.info("No available alive server from lb.getReachableServers(), failback to lb.getAllServers()");
			//如果找不到，从eureka拿到的所有UP状态的服务器中找一台
			servers = lb.getAllServers();
		}

		Server server = chooseFromServers(servers, getLoadBalancer(), key);

		if (server != null && server instanceof DiscoveryEnabledServer) {
			DiscoveryEnabledServer server1 = (DiscoveryEnabledServer) server;
			String isPerfTestSupportted = server1.getInstanceInfo().getMetadata().get(DubboPerfTestRegistryFactoryWrapper.IS_PERF_TEST_SUPPORTTED_KEY);
			if (!"1".equals(isPerfTestSupportted) && InternalPerfTestContext.isCurrentInPerfTestMode()) {
				throw new IllegalStateException("识别到当前请求是压测流量，但是调用的服务不支持性能压测，放弃本次请求，请通知该服务负责人加入spring-boot-starter-perftest依赖；尝试调用的服务端：" +
						server1.getInstanceInfo().getVIPAddress() +
						",ip:" + server1.getInstanceInfo().getIPAddr() +
						",port:" + server1.getInstanceInfo().getPort());
			}
		}

		return server;
	}

	protected abstract Server chooseFromServers(List<Server> servers, ILoadBalancer loadBalancer, Object key);

	/**
	 * 根据服务器启动时间计算服务器权重，启动时间越久，权重越大(预热时间5分钟，5分钟后weight会达到服务器设定的最大值)
	 * @param server
	 * @return
	 */
	protected int getTimeBasedWeight(Server server) {
		int weight = 100;
		if(server instanceof DiscoveryEnabledServer){
			String weightStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.WEIGHT_KEY);
			weight = NumberUtils.parseInt(weightStr, 100);
			weight = Math.max(weight, 0);
		}

		if (weight > 0) {
			//获取目标服务器启动时间
			long timestamp = 0;
			int warmUpTimeMillis = 300000;//预热时间，单位:ms
			if(server instanceof DiscoveryEnabledServer){
				String tsStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.SERVER_START_UP_TIME_KEY);
				timestamp = NumberUtils.parseLong(tsStr,0);
				timestamp = Math.max(timestamp, 0);

				String wuStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_WARMUP_TIMEMILLIS);
				warmUpTimeMillis = NumberUtils.parseInt(wuStr, 300000);
				warmUpTimeMillis = Math.max(warmUpTimeMillis, 60000);//最少也要60秒预热时间
			}

			if (timestamp > 0L) {
				int uptime = (int) (System.currentTimeMillis() - timestamp);
				if (uptime <= 0){
					weight = 1;
				}
				else if (uptime > 0 && uptime < warmUpTimeMillis) {
					weight = calculateWarmupWeight(uptime, warmUpTimeMillis, weight);
				}
			}
		}
		return weight;
	}

	private int calculateWarmupWeight(int uptime, int warmup, int weight) {
		int ww = (int) ( (float) uptime / ( (float) warmup / (float) weight ) );
		if(ww < 1){
			return 1;
		}
		return (ww > weight ? weight : ww);
	}
}
