package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 回放 RedisAtomicClient
 * Created by guoyanfei .
 * 2019-06-20 .
 */
@Slf4j
public class ReplayRedisAtomicClientMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {  // NOSONAR
        if (!canReplay(method, allArguments)) {
            return;
        }
        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        log.debug("RedisAtomicClient回放_traceId={}_spanType={}_spanId={}", ReplayTraceContext.getContextTraceId(), span != null ? span.getSpanType() : null, span != null ? span.getSpanId() : null);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.REDIS != span.getSpanType()) {
            String expert = SpanType.REDIS.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RedisFlowReplaySpan redisSpan = (RedisFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(redisSpan.getMethodName(), method)) {
            String expert = redisSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_102, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!FlowReplayUtils.isArgumentsTypesEqual(redisSpan.getParameterTypes(), argumentsTypes)) {
            String expert = FlowReplayUtils.stringArrayToString(redisSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(argumentsTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_100, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object[] recordParameterValues = redisSpan.getParameterValues();
            Object[] replayParameterValues = allArguments;
            if ("getLock".equals(method.getName())) {
                recordParameterValues = new Object[] { recordParameterValues[0] };
                replayParameterValues = new Object[] { replayParameterValues[0] };
            } else if ("incrBy".equals(method.getName())) {
                recordParameterValues = new Object[] { recordParameterValues[0], recordParameterValues[1] };
                replayParameterValues = new Object[] { replayParameterValues[0], replayParameterValues[1] };
            }
            // 参数值是否相同
            if (!FlowReplayUtils.isArgumentsEqual(recordParameterValues, replayParameterValues)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_101);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_103, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object ret = null;
            if ("getLock".equals(method.getName())) {
                boolean isLock = (boolean) redisSpan.getReturnValue();
                if (isLock) {
                    ret = new RedisLock() {

                        @Override
                        public void unlock() {
                            // no thing
                        }

                        @Override
                        public void close() throws Exception {
                            this.unlock();
                        }
                    };
                }
            } else {
                ret = redisSpan.getReturnValue();
            }
            result.defineReturnValue(ret);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_104, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // do nothing
    }
}
