package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.utils.MainApplicationContextHolder;
import com.netflix.client.config.IClientConfig;
import com.netflix.loadbalancer.AbstractLoadBalancerRule;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.IRule;
import com.netflix.loadbalancer.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.web.context.ConfigurableWebApplicationContext;

/**
 * 负载均衡规则，支持根据配置动态更改规则
 *
 * @author wenqi.huang
 */
public class RuntimeRule extends AbstractLoadBalancerRule implements ApplicationListener<EnvironmentChangeEvent> {

    private static Logger log = LoggerFactory.getLogger(RuntimeRule.class);

    private static final String RIBBON_LOADBALANCE_TYPE_KEY = "duiba.ribbon.loadbalance.type";

    //真正的规则
    private volatile IRule delegate;

    private IClientConfig clientConfig;

    //服务过滤的filter列表，用于运行时过滤ribbon服务器
    private List<RibbonServerListFilter> ribbonServerListFilters = Collections.emptyList();

    @Autowired(required=false)
    public void setRibbonServerListFilters(List<RibbonServerListFilter> ribbonServerListFilters){
        if(ribbonServerListFilters != null) {
            this.ribbonServerListFilters = ribbonServerListFilters;
        }
    }

    @Resource
    public ApplicationContext applicationContext;

    @PostConstruct
    public void init(){
    	//这里注入的applicationContext有可能是ribbon所在的子ApplicationContext，无法获取到刷新后的环境变量，且注册ApplicationListener不会生效，所以需要获取有效的
        ApplicationContext tempContext = applicationContext;
        while(!(tempContext instanceof ConfigurableWebApplicationContext)){
            tempContext = applicationContext.getParent();
        }
        if(tempContext instanceof ConfigurableWebApplicationContext) {
            applicationContext = tempContext;
        }

        String loadBalanceType = applicationContext.getEnvironment().getProperty(RIBBON_LOADBALANCE_TYPE_KEY, "weightedRoundRobin");
        FilterBasedRule temp;
        if ("weightedRoundRobin".equals(loadBalanceType)){
            temp = new WeightedRoundRobinRule(this.getLoadBalancer(), ribbonServerListFilters);
        }else if("weightedRandom".equals(loadBalanceType)){
            temp = new WeightedRandomRule(this.getLoadBalancer(), ribbonServerListFilters);
        }else if("weightedLeastConnection".equals(loadBalanceType) || "leastactive".equals(loadBalanceType)){//leastactive废弃
            temp = new WeightedLeastConnectionRule(this.getLoadBalancer(), ribbonServerListFilters);
        }else{
            temp = new WeightedRoundRobinRule(this.getLoadBalancer(), ribbonServerListFilters);
        }

        temp.initWithNiwsConfig(clientConfig);
        delegate = temp;

        if(applicationContext instanceof ConfigurableWebApplicationContext) {
            ((ConfigurableWebApplicationContext)tempContext).addApplicationListener(this);
        }else{
            log.warn("Not found ConfigurableWebApplicationContext, {} will no be refreshable", RIBBON_LOADBALANCE_TYPE_KEY);
        }
    }

    @Override
    public void onApplicationEvent(EnvironmentChangeEvent event) {
        if(event.getKeys() == null || !event.getKeys().contains(RIBBON_LOADBALANCE_TYPE_KEY)){
            return;
        }

        //如果检测到刷新了负载均衡算法，则变更负载均衡实例
        init();
    }

    @Override
    public Server choose(Object key) {
        Server server = delegate.choose(key);

        return server;
    }

    @Override
    public void initWithNiwsConfig(IClientConfig clientConfig) {
        this.clientConfig = clientConfig;
    }

    @Override
    public void setLoadBalancer(ILoadBalancer lb) {
        super.setLoadBalancer(lb);
        if(delegate == null){
            //如果delegate为null，说明是ZoneAwareLoadBalancer clone出来的，需要把相关的属性设置上！
            this.applicationContext = MainApplicationContextHolder.getApplicationContext();
            if(applicationContext == null){
                log.warn("[NOTIFYME]applicationContext is null, will not be here");
                return;
            }
            this.ribbonServerListFilters =
                    new ArrayList<>(MainApplicationContextHolder.getApplicationContext().getBeansOfType(RibbonServerListFilter.class).values());
            init();
        }

        this.delegate.setLoadBalancer(lb);
    }

}
