package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import java.util.List;

import cn.com.duiba.boot.concurrent.AtomicPositiveInteger;
import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;
import org.apache.commons.lang3.RandomUtils;

/**
 * 加权轮询 负载均衡规则
 * //TODO 这里的算法不太均衡，按这个文章改进：https://blog.csdn.net/larry_zeng1/article/details/80407745
 *
 * @author wenqi.huang
 */
public class WeightedRoundRobinRule extends FilterBasedRule {

    //初始值赋予一个随机数，以防止所有服务器初始时的调用序列一致，而使某些服务器流量超高
    private final AtomicPositiveInteger nextIndex
            = new AtomicPositiveInteger(RandomUtils.nextInt(0, Integer.MAX_VALUE));

    public WeightedRoundRobinRule(ILoadBalancer lb, List<RibbonServerListFilter> serverListFilters) {
        super(serverListFilters);
        setLoadBalancer(lb);
    }

    @Override
    protected Server chooseFromServers(List<Server> servers, ILoadBalancer lb, Object key) {
        Server server = null;
        int count = 0;
        int maxFindTimes = Math.min(10, servers.size() + 1);
        while (server == null && count++ < maxFindTimes) {
            server = chooseRoundRobinWithTimeBasedWeight(servers);

            if (server == null) {
                /* Transient. */
                Thread.yield();
                continue;
            }

            if (server.isAlive() && (server.isReadyToServe())) {
                return (server);
            }

            // Next.
            server = null;
        }

        if (count >= 10) {
            log.warn("No available alive servers after 10 tries from load balancer: "
                    + lb);
        }
        return server;
    }

    /**
     * 使用权重加权轮询算法获取一个服务器。其中权重根据服务器启动时间变化，以达到对于刚启动的服务器逐渐增加流量的目的
     * @param eligible
     * @return
     */
    private Server chooseRoundRobinWithTimeBasedWeight(List<Server> eligible){
        int length = eligible.size(); // 总个数
        int totalWeight = 0; // 总权重
        boolean sameWeight = true; // 权重是否都一样
        int[] weights = new int[length];
        for (int i = 0; i < length; i++) {
            int weight = getTimeBasedWeight(eligible.get(i));
            weights[i] = weight;
            totalWeight += weight; // 累计总权重
            if (sameWeight && i > 0
                    && weight != weights[i - 1]) {
                sameWeight = false; // 计算所有权重是否一样
            }
        }
        if (totalWeight > 0 && ! sameWeight) {
            // 如果权重不相同且权重大于0则按总权重数轮询
            int offset = nextIndex.getAndIncrement() % totalWeight;
            // 并确定下一个值落在哪个片断上
            for (int i = 0; i < length; i++) {
                offset -= weights[i];
                if (offset < 0) {
                    return eligible.get(i);
                }
            }
        }

        // 如果权重相同或权重为0则直接轮询即可
        return eligible.get(nextIndex.getAndIncrement() % length);
    }

}
