package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duiba.boot.utils.MainApplicationContextHolder;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayConfigDto;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayEndEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayStartEvent;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;

import java.lang.management.ManagementFactory;
import java.util.Arrays;
import java.util.List;

/**
 * 实例的回放上下文 持有者
 * Created by guoyanfei .
 * 2019-02-26 .
 */
public class ReplayContextHolder {

    private static final List<String> DEFAULT_HOST_WHITELIST = Arrays.asList("127.0.0.1", "localhost");

    private ReplayContextHolder() {
    }

    private static volatile boolean replaying = false;

    private static volatile ReplayContext replayContext;

    /**
     * 回放机器能访问的ip白名单
     */
    private static volatile List<String> ipWhitelist;

    /**
     * 回放的时候，需要把系统当前时间设置成录制的时候的时间
     * 本字段在回放开始的时候，会设置成【录制开始时间】
     * 每回放1分钟的用例（指的是录制1分钟内的用例，非回放持续时间）都会把本字段更新成下一分钟的第一个用例的时间
     * 这样可以尽可能的模拟录制的用例在当时请求的时候的系统时间，让引流回归平台能尽可能覆盖到大部分时间敏感用例
     */
    private static volatile long mockSysTime;

    /**
     * 用例集的录制开始时间
     * 用于计算每个用例在实际录制过程中的请求时间
     */
    private static volatile long recordStartTime;

    /**
     * 记录单次回放的排错信息，用于回放后问题的排查
     */
    private static volatile JSONObject debugInfo;

    /**
     * 回放操作被屏蔽的接口列表
     * 用户可以在引流回归系统后台把一部分接口屏蔽掉，屏蔽掉的接口不参与回放操作
     */
    private static volatile List<String> maskedApis;

    /**
     * 回放操作被屏蔽的用例列表
     * 预回放结果中的一些失败用例，在正式回放中会被自动忽略掉，此处目前只有此类用例
     */
    private static volatile List<String> maskedTraceIds;

    /**
     * 当前实例是否在回放中
     * @return
     */
    public static boolean isReplaying() {
        return replaying && (replayContext != null);
    }

    /**
     * 开始回放
     * @param replayConfig
     */
    public synchronized static void start(ReplayConfigDto replayConfig) {
        if (!replaying) {
            // 回放标记为开始
            replaying = true;

            // 初始化回放配置
            replayContext = new ReplayContext(replayConfig);

            // 设置ip白名单，除了这部分ip，回放机器不能访问其他ip
            ipWhitelist = replayConfig.getIpWhitelist();
            ipWhitelist.addAll(DEFAULT_HOST_WHITELIST);

            // 设置回放屏蔽接口
            maskedApis = replayConfig.getMaskedApis();

            // 设置回放屏蔽用例
            maskedTraceIds = replayConfig.getMaskedTraceIds();

            // 回放开始
            replayContext.start();

            // mock回放机器的系统时间为用例集的开始录制时间，尽量减少一些时间敏感用例对比不一致或者失败的情况
            recordStartTime = replayConfig.getStartTime();
            mockSysTime = replayConfig.getStartTime();
            ReplayMockSupport.setOffsetMillis(mockSysTime - replayContext.getStartTime());

            // 设置debug信息，用于返回引流回归系统服务端，排错用
            debugInfo = new JSONObject();
            debugInfo.put("mockSysTime", DateUtils.getSecondStr(System.currentTimeMillis()));
            debugInfo.put("jvmArgs", ManagementFactory.getRuntimeMXBean().getInputArguments());

            // 发布回放开始事件
            MainApplicationContextHolder.getApplicationContext().publishEvent(new ReplayStartEvent());
        } else {
            throw new FlowReplayException("回归操作已经开始_请勿重复开始");
        }
    }

    /**
     * 获取debugInfo返回给引流回归系统服务端
     * @return
     */
    public static String debugInfo() {
        if (debugInfo != null) {
            return debugInfo.toJSONString();
        }
        return null;
    }

    /**
     * 判断回放容器是否可以访问指定的host
     * @param host
     * @return
     */
    public static boolean canHostPass(String host) {
        // 如果ip白名单还没设置，那么所有host都都可以正常访问
        if (CollectionUtils.isEmpty(ipWhitelist)) {
            return true;
        }
        // 如果ip白名单已经设置，那么值允许ip白名单的可以访问
        return ipWhitelist.contains(host);
    }

    /**
     * 判断当前待回放的接口是否在被屏蔽的接口列表中
     * @param apiName
     * @return
     */
    public static boolean isMaskedApi(String apiName) {
        List<String> maskedApisTemp = maskedApis;
        if (CollectionUtils.isEmpty(maskedApisTemp)) {
            return false;
        }
        return maskedApisTemp.contains(apiName);
    }

    /**
     * 判断当前待回放的用例是否在被屏蔽的用例列表中
     * @param traceId
     * @return
     */
    public static boolean isMaskedTraceIds(String traceId) {
        List<String> maskedTraceIdsTemp = maskedTraceIds;
        if (CollectionUtils.isEmpty(maskedTraceIdsTemp)) {
            return false;
        }
        return maskedTraceIdsTemp.contains(traceId);
    }

    /**
     * 计算每个用例在实际录制过程中的请求时间
     * @param secondsAfterStart
     * @return
     */
    public static long calculateTraceTimestamp(short secondsAfterStart) {
        return recordStartTime + (secondsAfterStart * 1000);
    }

    /**
     * 回放过程中，通过判断当前用例的录制时间，来确定是否需要把系统当前时间mock成这个用例的录制时间
     * 详细逻辑见 mockSysTime 字段的说明
     * @param traceTimestamp
     * @return
     */
    public static boolean isMockSysTimeValid(long traceTimestamp) {
        return (traceTimestamp - mockSysTime) < 60 * 1000;
    }

    /**
     * 把当前系统时间设置成回放用例的时间
     * 详细逻辑见 mockSysTime 字段的说明
     * @param traceTimestamp
     */
    public synchronized static void setMockSysTime(long traceTimestamp) {
        if (isMockSysTimeValid(traceTimestamp)) {
            return;
        }
        mockSysTime = traceTimestamp;
        ReplayMockSupport.setOffsetMillis(mockSysTime - replayContext.getStartTime());
    }

    /**
     * 结束回放
     */
    public synchronized static void end() {
        if (replaying) {
            replaying = false;
            ReplayMockSupport.setOffsetMillis(0);
            replayContext.end();
            MainApplicationContextHolder.getApplicationContext().publishEvent(new ReplayEndEvent());
        } else {
            throw new FlowReplayException("回归操作已经结束_请勿重复结束");
        }
    }

    /**
     * 强制结束回放
     */
    public static void forceEnd() {
        replayContext.tracesLoadFinish();
        replayContext.replayersDone();
        end();
    }

    /**
     * 获取当前实例的回放上下文
     * @return
     */
    public static ReplayContext getReplayContext() {
        return replayContext;
    }

}
