package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import com.alibaba.fastjson.JSON;
import com.google.common.cache.Cache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.ehcache.PersistentUserManagedCache;
import org.ehcache.config.builders.CacheEventListenerConfigurationBuilder;
import org.ehcache.config.builders.ResourcePoolsBuilder;
import org.ehcache.config.builders.UserManagedCacheBuilder;
import org.ehcache.config.units.MemoryUnit;
import org.ehcache.core.spi.service.LocalPersistenceService;
import org.ehcache.event.CacheEventListener;
import org.ehcache.event.EventType;
import org.ehcache.impl.config.persistence.DefaultPersistenceConfiguration;
import org.ehcache.impl.config.persistence.UserManagedPersistenceContext;
import org.ehcache.impl.persistence.DefaultLocalPersistenceService;
import org.springframework.core.env.Environment;

import java.io.File;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;


/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-07-24 15:44
 * @descript:
 * @version: 1.0
 */
@Slf4j
class CacheMonitorManager {

    private static Environment environment;

    private static final String localIp = NetUtils.getLocalIp();

    private static String currentAppName;

    private static final ConcurrentMap<Object, String> allCacheMap = Maps.newConcurrentMap();

    private static final ConcurrentMap<Integer, Object> cacheBuilderMap = Maps.newConcurrentMap();

    private static final ArrayBlockingQueue<CacheChangeRecord> recordQueue = new ArrayBlockingQueue<>(1000);

    private static final ConcurrentMap<String, CopyOnWriteArrayList<Integer>> cachePageUtilMap = Maps.newConcurrentMap();

    private static final int pageSize = 30;

    private static boolean isConsuming = true;

    static volatile boolean isOpenRecord = false;

    // 保存本地缓存的变化
    private static PersistentUserManagedCache<String, ArrayList> monitorCache = null;

    static void saveCache(Object key, String value) {
        allCacheMap.put(key, value);
    }

    static Set<String> getAllCache() {

        Set<String> names = Sets.newHashSet();
        for (Map.Entry<Object, String> entry : allCacheMap.entrySet()) {
            names.add(entry.getValue());
        }

        return names;
    }

    static Object getCacheValueByKey(String cacheName, String key) {

        Object cacheObj = null;
        for (Map.Entry<Object, String> entry : allCacheMap.entrySet()) {

            if (entry.getValue().equals(cacheName)) {
                cacheObj = entry.getKey();
                break;
            }
        }

        if (cacheObj == null) {
            return null;
        }

        if (cacheObj instanceof Cache) {

            Cache cache = (Cache) cacheObj;
            Iterator iterator = cache.asMap().keySet().iterator();
            if (!iterator.hasNext()) {
                return null;
            }

            Class aClass = iterator.next().getClass();
            return cache.getIfPresent(JSON.parseObject(key, aClass));
        }

        if (cacheObj instanceof com.github.benmanes.caffeine.cache.Cache) {
            com.github.benmanes.caffeine.cache.Cache cache = (com.github.benmanes.caffeine.cache.Cache) cacheObj;
            Iterator iterator = cache.asMap().keySet().iterator();
            if (!iterator.hasNext()) {
                return null;
            }

            Class aClass = iterator.next().getClass();
            return cache.getIfPresent(JSON.parseObject(key, aClass));
        }

        return null;
    }

    static RecordPagerResponse<CacheChangeRecord> getRecord(String cacheName, String key, Integer page) {

        if (StringUtils.isBlank(cacheName) || StringUtils.isBlank(key) || monitorCache == null) {
            return null;
        }

        if (page == null) {
            page = 1;
        }

        String cacheKey = String.format("%s-%s", cacheName, key);
        CopyOnWriteArrayList<Integer> pageUtils = cachePageUtilMap.get(cacheKey);
        if (pageUtils == null) {
            return null;
        }

        Collections.reverse(pageUtils);// 倒序
        Integer pageIndex = pageUtils.get(page - 1);
        String monitorKey = String.format("%s-%s-%s", cacheName, key, pageIndex);
        List<CacheChangeRecord> values = monitorCache.get(monitorKey);
        if (CollectionUtils.isEmpty(values)) {
            return null;
        }

        Collections.reverse(values);
        int total = (pageUtils.size() - 1) * pageSize + values.size();
        return new RecordPagerResponse(page, pageSize, total, values);
    }

    static void setBuilderCache(Integer hashcode, Object value) {
        cacheBuilderMap.put(hashcode, value);
    }

    /**
     * 使用之前一定需要isOpenRecord判断
     *
     * @param record
     */
    static void addRecord(CacheChangeRecord record) {

        Object cache = cacheBuilderMap.get(record.getBuilderHashcode());
        if (cache == null) {
            return;
        }

        String cacheName = allCacheMap.get(cache);
        if (StringUtils.isBlank(cacheName)) {
            return;
        }

        record.setCacheName(cacheName);
        recordQueue.offer(record);
    }

    static void init(Environment env, String appName, int httpServerPort) {

        if (env != null) {
            environment = env;
        } else {
            log.warn("缓存监控示例未初始化environment，无法监控缓存的变化");
        }

        if (appName != null) {
            currentAppName = appName;
            LocalPersistenceService persistenceService = new DefaultLocalPersistenceService(new DefaultPersistenceConfiguration(new File(System.getProperty("user.home"), String.format("/cache_monitor/%s/%s", httpServerPort, appName))));
            monitorCache = UserManagedCacheBuilder.newUserManagedCacheBuilder(String.class, ArrayList.class)
                    .withEventExecutors(Executors.newSingleThreadExecutor(new NamedThreadFactory("cache-monitor-ordered", true)), Executors.newSingleThreadExecutor(new NamedThreadFactory("cache-monitor-unordered", true)))
                    .withEventListeners(CacheEventListenerConfigurationBuilder
                            .newEventListenerConfiguration((CacheEventListener) event -> {

                                String key = String.valueOf(event.getKey());
                                String cacheKey = key.substring(0, key.lastIndexOf("-"));
                                Integer curPage = Integer.parseInt(key.substring(key.lastIndexOf("-") + 1));
                                CopyOnWriteArrayList<Integer> pageUtil = cachePageUtilMap.get(cacheKey);
                                if (pageUtil == null) {
                                    return;
                                }

                                pageUtil.remove(curPage);

                                if (pageUtil.size() == 0) {
                                    cachePageUtilMap.remove(cacheKey);
                                }

                            }, EventType.REMOVED, EventType.EVICTED, EventType.EXPIRED)
                            .asynchronous()
                            .unordered())
                    .with(new UserManagedPersistenceContext<>("monitorCache", persistenceService)).withResourcePools(ResourcePoolsBuilder.newResourcePoolsBuilder()
                            .heap(1, MemoryUnit.MB)//堆内存最大占用
                            .disk(100L, MemoryUnit.MB, true))//磁盘占用量
                    .build(true);
        } else {
            log.warn("缓存监控示例未初始化当前系统名称，无法监控缓存的变化");
        }

        changeOpenRecord();


        startConsume();
    }

    static void changeOpenRecord() {

        if (environment != null && StringUtils.isNotBlank(currentAppName) && isConsuming) {
            String openIp = environment.getProperty(CacheMonitorConstant.RECORD_IP_CONFIG_KEY);
            isOpenRecord = StringUtils.equals(localIp, openIp);
        }
    }

    private static void startConsume() { //NOSONAR

        if (monitorCache == null) {
            return;
        }

        Thread consume = new Thread(() -> {

            while (isConsuming) {

                try {

                    CacheChangeRecord newRecord = recordQueue.take();
                    if (StringUtils.isBlank(newRecord.getCacheName())) {
                        continue;
                    }

                    String pageKey = getPageUtilCacheKey(newRecord.getCacheName(), String.valueOf(newRecord.getKey()));
                    CopyOnWriteArrayList<Integer> pageUtil = cachePageUtilMap.get(pageKey);
                    if (pageUtil == null) {
                        pageUtil = Lists.newCopyOnWriteArrayList(Lists.newArrayList(1));
                        cachePageUtilMap.put(pageKey, pageUtil);
                    }

                    int curPage = pageUtil.get(pageUtil.size() - 1);
                    String monitorKey = getMonitorCacheKey(newRecord.getCacheName(), String.valueOf(newRecord.getKey()), curPage);

                    ArrayList oldRecords = monitorCache.get(monitorKey);
                    if (CollectionUtils.isEmpty(oldRecords)) {
                        oldRecords = Lists.newArrayList();
                        newRecord.buildSerializableObj();
                        oldRecords.add(newRecord);
                        monitorCache.put(monitorKey, oldRecords);
                        continue;
                    }

                    CacheChangeRecord lastRecord = (CacheChangeRecord) oldRecords.get(oldRecords.size() - 1);
                    if (newRecord.equals(lastRecord)) {
                        continue;
                    }

                    newRecord.buildSerializableObj();
                    if (oldRecords.size() < pageSize) {
                        oldRecords.add(newRecord);
                        monitorCache.put(monitorKey, oldRecords);
                    } else {
                        curPage++;
                        pageUtil.add(curPage);
                        monitorKey = getMonitorCacheKey(newRecord.getCacheName(), String.valueOf(newRecord.getKey()), curPage);
                        monitorCache.put(monitorKey, Lists.newArrayList(newRecord));
                    }


                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } catch (Throwable throwable) {
                    log.error("消费缓存的监控异常", throwable);
                }

                if (Thread.currentThread().isInterrupted()) {
                    isConsuming = false;
                }
            }

        }, "cache-monitor-consume");

        consume.setDaemon(true);
        consume.start();
    }

    private static String getMonitorCacheKey(String cacheName, String key, Integer page) {
        return String.format("%s-%s-%s", cacheName, key, page);
    }

    private static String getPageUtilCacheKey(String cacheName, String key) {
        return String.format("%s-%s", cacheName, key);
    }
}


