package cn.com.duiba.boot.ext.autoconfigure.cloud.netflix.feign.hystrix;

import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixCommand;
import cn.com.duiba.boot.netflix.feign.hystrix.FeignHystrixProperty;
import cn.com.duiba.boot.netflix.feign.hystrix.conf.HystrixPropertiesManager;
import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixCommandGroupKey;
import com.netflix.hystrix.HystrixCommandKey;
import com.netflix.hystrix.HystrixThreadPoolKey;
import feign.Feign;
import feign.RetryableException;
import feign.Retryer;
import feign.Target;
import feign.codec.ErrorDecoder;
import feign.hystrix.HystrixFeign;
import feign.hystrix.SetterFactory;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.RejectedExecutionException;

/**
 * Feign客户端自定义，覆盖 FeignClientsConfiguration 中对应配置
 */
@Configuration
@ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
public class HystrixFeignConfiguration {

	/**
	 * 使用自定义的SetterFactory,以获得在代码中自定义超时时间、自定义熔断器配置的特性
	 * @return
	 */
	@Bean
	@Scope("prototype")
	@ConditionalOnMissingBean
	@ConditionalOnProperty(name = "feign.hystrix.enabled", matchIfMissing = true)
	public Feign.Builder feignHystrixBuilder() {
		return HystrixFeign.builder().setterFactory(new HystrixFeignSetterFactory());
	}

	/**
	 * 自定义Feign错误解码器，把非2**状态的http响应转换为异常。对于IllegalArgumentException或者注解了@HystrixDisabled的异常会包装为HystrixBadRequestException，避免触发熔断/降级
	 * @return
	 */
	@Bean
	public ErrorDecoder feignErrorDecoder(){
		return new FeignErrorDecoder();
	}

	/**
	 * 自定义重试处理器,遇到RejectedExecutionException进行重试。(这里不处理ConnectException，因为Ribbon内部的重试处理器会对这个进行处理进行重试, 为了避免重复处理，这里不需要处理ConnectionException）
	 */
	@Configuration
	public static class CustomRetryer implements Retryer {
		private int maxAttempts = 2;
		private int attempt = 1;

		@Override
		public void continueOrPropagate(RetryableException e) {
			if(attempt++ >= maxAttempts){
				throw e;
			}
			Throwable cause = e.getCause();
			if(cause == null) {
				throw e;
			}
			else if(cause instanceof RejectedExecutionException){
				//RejectedExecutionException表示服务端线程池满了或者繁忙或者其他原因拒绝,50ms后重试1次
				try {
					Thread.sleep(50);
				} catch (InterruptedException e1) {
					Thread.currentThread().interrupt();
				}
			}else{
				throw e;
			}
		}

		@Override
		public Retryer clone() {
			return new CustomRetryer();
		}
	}

	/**
	 * 这个类会自动识别出FeignClient(类、方法)上注解了@FeignHystrixProperty/@FeignHystrixCommand的方法，并应用其中的属性，比如超时时间等
	 */
	final class HystrixFeignSetterFactory implements SetterFactory {

		@Override
		public HystrixCommand.Setter create(Target<?> target, Method method) {
			String defaultGroupKey = target.name();
			String defaultCommandKey = Feign.configKey(target.type(), method);

			FeignHystrixCommand methodCommand = method.getAnnotation(FeignHystrixCommand.class);
			FeignHystrixProperty[] methodProperties = method.getAnnotationsByType(FeignHystrixProperty.class);
			FeignHystrixCommand classCommand = target.type().getAnnotation(FeignHystrixCommand.class);
			FeignHystrixProperty[] classProperties = target.type().getAnnotationsByType(FeignHystrixProperty.class);
			List<FeignHystrixCommand> commands = new ArrayList<>();
			commands.add(methodCommand);
			commands.add(classCommand);
			String groupKey = commands.stream().filter((c) -> c!=null && StringUtils.isNotBlank(c.groupKey())).findFirst().map(c -> c.groupKey()).orElse(defaultGroupKey);
			String commandKey = methodCommand == null || StringUtils.isBlank(methodCommand.commandKey()) ? defaultCommandKey : methodCommand.commandKey();
			String threadPoolKey = commands.stream().filter((c) -> c!=null && StringUtils.isNotBlank(c.threadPoolKey())).findFirst().map(c -> c.threadPoolKey()).orElse(null);

			HystrixCommand.Setter setter = HystrixCommand.Setter.withGroupKey(HystrixCommandGroupKey.Factory.asKey(groupKey));
			setter.andCommandKey(HystrixCommandKey.Factory.asKey(commandKey));
			if(threadPoolKey != null) {
				setter.andThreadPoolKey(HystrixThreadPoolKey.Factory.asKey(threadPoolKey));
			}

			//构造属性list，最低优先级的放在最前，以让后面的可以覆盖前面的配置
			List<FeignHystrixProperty> allProperties = new ArrayList<>();
			if(classCommand != null && !ArrayUtils.isEmpty(classCommand.properties())){
				for(FeignHystrixProperty p : classCommand.properties()) {
					allProperties.add(p);
				}
			}
			if(!ArrayUtils.isEmpty(classProperties)){
				for(FeignHystrixProperty p : classProperties) {
					allProperties.add(p);
				}
			}
			if(methodCommand != null && !ArrayUtils.isEmpty(methodCommand.properties())){
				for(FeignHystrixProperty p : methodCommand.properties()) {
					allProperties.add(p);
				}
			}
			if(!ArrayUtils.isEmpty(methodProperties)){
				for(FeignHystrixProperty p : methodProperties) {
					allProperties.add(p);
				}
			}

			setter.andCommandPropertiesDefaults(HystrixPropertiesManager.initializeCommandProperties(allProperties));
			setter.andThreadPoolPropertiesDefaults(HystrixPropertiesManager.initializeThreadPoolProperties(allProperties));

			return setter;
		}
	}
}
