package cn.com.duiba.boot.ext.autoconfigure.initserver;

import cn.com.duiba.boot.ext.autoconfigure.core.AsyncSpecifiedBeanPostProcessor;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.web.context.WebApplicationContext;

import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by wenqi.huang on 2017/6/16.
 */
public class ApplicationListenerForAsyncSpecified implements ApplicationListener<ContextRefreshedEvent>, Ordered {
    private Map<AsyncSpecifiedBeanPostProcessor, Map<String, Object>> asyncProcessor2BeansMap;

    public ApplicationListenerForAsyncSpecified(Map<AsyncSpecifiedBeanPostProcessor, Map<String, Object>> asyncProcessor2BeansMap) {
        this.asyncProcessor2BeansMap = asyncProcessor2BeansMap;
    }

    //指定顺序，让当前的ApplicationListener被优先执行（比Dubbo暴露服务先执行）
    @Override
    public int getOrder() {
        return -2;
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        ApplicationContext context = event.getApplicationContext();
        //EmbeddedWebApplicationContext
        if(!(context instanceof WebApplicationContext)){//只对WebApplicationContext进行初始化，不对子context初始化，以防发生死锁
            return;
        }

        //开启线程池用于异步方法，执行完成后立即销毁
        ExecutorService executorService = Executors.newFixedThreadPool(100, new NamedThreadFactory("DuibaBootInitServerThread-"));

        //异步化执行所有AsyncSpecifiedBeanPostProcessor
        int totalBean2ProcessCount = 0;
        for(Map<String, Object> beanMap : asyncProcessor2BeansMap.values()){
            totalBean2ProcessCount += beanMap.size();
        }

        final CountDownLatch countDownLatch = new CountDownLatch(totalBean2ProcessCount);

        for(Map.Entry<AsyncSpecifiedBeanPostProcessor, Map<String, Object>> entry : asyncProcessor2BeansMap.entrySet()){
            Map<String, Object> beanMap = entry.getValue();
            final AsyncSpecifiedBeanPostProcessor processor = entry.getKey();
            for(final Map.Entry<String, Object> beanEntry : beanMap.entrySet()){
                executorService.submit(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            processor.postProcessAfterInitialization(beanEntry.getValue(), beanEntry.getKey());
                        }finally {
                            countDownLatch.countDown();
                        }
                    }
                });
            }
        }

        //阻塞住spring流程，确保异步初始化全部执行完成再继续
        try {
            countDownLatch.await();
        }catch(InterruptedException e){
            //Ignore
        }

        executorService.shutdown();
    }
}
