package cn.com.duiba.boot.ext.autoconfigure.jackson;

import cn.com.duiba.boot.ext.autoconfigure.core.DuibaServerProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

import javax.annotation.Resource;

/**
 * jackson自定义，具体内容参考下面的注释
 */
@Configuration
public class JacksonCustomAutoConfiguration {

    /**
     * 声明ProxyedClassJacksonModule，这个Module会被JacksonAutoConfiguration自动找到并注入到jaskcon的module中，指定的类会优先使用这个module来序列化
     * 这个module主要用于修复经过代理的类序列化时会输出多余属性(或者抛异常)的问题。指定不输出的属性是写死的，如果被代理对象有相同属性，也不会输出，不过属性重名的可能性非常小。
     */
    @Configuration
    @ConditionalOnClass({ ObjectMapper.class })
    public static class ProxyedClassJacksonModuleConfiguration{
        @Bean
        public ProxyedClassJacksonModule proxyedClassJacksonModuleConfigurer(){
            return new ProxyedClassJacksonModule();
        }
    }

    /**
     * 自定义jackson,设置jackson反序列化时遇到未知属性不报错<br/>
     * 如果是内部服务，则设置jackson序列化时遇到null属性不序列化，即对应的key不会输出
     */
    @Configuration
    @EnableConfigurationProperties(DuibaServerProperties.class)
    public static class CustomJackson2ObjectMapperBuilderCustomizer implements Jackson2ObjectMapperBuilderCustomizer,Ordered {

        @Resource
        private DuibaServerProperties duibaServerProperties;
        @Override
        public void customize(Jackson2ObjectMapperBuilder jacksonObjectMapperBuilder) {
            //设置jackson反序列化时遇到未知属性不报错
            jacksonObjectMapperBuilder.failOnUnknownProperties(false);

            //如果是内部服务，则设置jackson序列化时遇到null属性不序列化，即对应的key不会输出
            if(duibaServerProperties.isInternalMode()){
                jacksonObjectMapperBuilder.serializationInclusion(JsonInclude.Include.NON_NULL);
            }
        }

        @Override
        public int getOrder() {
            return -1;//比JacksonAutoConfiguration.StandardJackson2ObjectMapperBuilderCustomizer先执行，以让property配置的优先级更高
        }
    }
}
