package cn.com.duibaboot.ext.autoconfigure.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.BeanCreationNotAllowedException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.SimpleApplicationEventMulticaster;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.core.ResolvableType;

/**
 * 配置自定义的ApplicationEventMulticaster广播器，加入ErrorHandler错误处理器，以修复部分spring的bug。
 * 启动时这个广播器没有配置线程池，事件仍然按顺序串行处理，不影响原先流程，此时错误处理器不会生效。
 * 在spring容器关闭前，GracefulCloseRunListener会给这个广播器加入线程池，让所有事件并行处理（关闭时事件处理顺序无所谓），并让错误处理器生效，从而有机会拦截spring的一个异常（spring的bug）。
 */
@Configuration
//@ConditionalOnProperty(name="spring.event.use-executor", havingValue = "true", matchIfMissing = true)
public class ApplicationEventMulticasterAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(ApplicationEventMulticasterAutoConfiguration.class);

    @Bean(name= AbstractApplicationContext.APPLICATION_EVENT_MULTICASTER_BEAN_NAME)
    public SimpleApplicationEventMulticaster simpleApplicationEventMulticaster(){
        CustomApplicationEventMulticaster multicaster = new CustomApplicationEventMulticaster();
        multicaster.setErrorHandler(t -> {
            if(t instanceof BeanCreationNotAllowedException){
                //修复spring的bug，忽略此异常；这个异常会在关闭时报出，原因是主容器销毁时，先销毁了eurekaAutoServiceRegistration这个bean，然后开始销毁feign子容器，子容器销毁时又发出了ContextClosedEvent, 这个事件会传播到主容器，主容器还保留着对eurekaAutoServiceRegistration里listener的引用，调用listener时尝试获取eurekaAutoServiceRegistration，而这个bean已经销毁，会尝试重建，而容器在销毁时不允许新建bean，所以报错
                if("eurekaAutoServiceRegistration".equals(((BeanCreationNotAllowedException)t).getBeanName())
                        && t.getMessage() != null
                        && t.getMessage().contains("Singleton bean creation not allowed while singletons of this factory are in destruction")){
                    return;
                }
            }
            logger.warn("Error calling ApplicationEventListener", t);
        });
        //这里不能设置线程池，GracefulCloseRunListener中会在spring关闭前设置线程池
//        multicaster.setTaskExecutor(TtlExecutors.getTtlExecutorService(new ThreadPoolExecutor(1, 5, 10, TimeUnit.SECONDS, new LinkedBlockingQueue<>())));
        return multicaster;
    }

    public static class CustomApplicationEventMulticaster extends SimpleApplicationEventMulticaster {

        private BeanFactory beanFactory;

        private boolean started = false;

        @Override
        public void setBeanFactory(BeanFactory beanFactory) {
            this.beanFactory = beanFactory;
            super.setBeanFactory(beanFactory);
        }

        @Override
        public void multicastEvent(ApplicationEvent event) {
            super.multicastEvent(event);
        }

        @Override
        public void multicastEvent(ApplicationEvent event, ResolvableType eventType) {
             if(event instanceof ContextRefreshedEvent
                    && event.getSource() != null
                    && event.getSource() instanceof AbstractApplicationContext){
                if (((AbstractApplicationContext) event.getSource()).getBeanFactory() == beanFactory) {
                    started = true;//主容器的ContextRefreshedEvent表示已经几乎启动完成
                } else if(!started) {
                    //在主容器启动完成前不把 子容器中的ContextRefreshedEvent传播到主容器,以防止一些bug：（在主容器根据a、b、c的顺序（前者依赖后者，c是一个FeignClient）创建bean的时候，当b创建到一半开始注入c时，需要先创建c，而创建c会开启一个子容器，子容器创建完成会发出ContextRefreshedEvent，这个事件也会传播到主容器，此时如果d恰好实现了ApplicationListener且d依赖了b，则会按d、b的顺序创建bean，当创建到b时发现之前b已经建立到了一般，则会抛出异常）,异常为：
                    //org.springframework.beans.factory.BeanCurrentlyInCreationException: Error creating bean with name 'creditsHdtoolOrdersDaoImpl': Bean with name 'creditsHdtoolOrdersDaoImpl' has been injected into other beans [creditsHdtoolOrdersServiceImpl] in its raw version as part of a circular reference, but has eventually been wrapped. This means that said other beans do not use the final version of the bean. This is often the result of over-eager type matching - consider using 'getBeanNamesOfType' with the 'allowEagerInit' flag turned off, for example.
                    //        at org.springframework.beans.factory.support.AbstractAutowireCapableBeanFactory.doCreateBean(AbstractAutowireCapableBeanFactory.java:585)
                    return;
                }
            }
            super.multicastEvent(event, eventType);
        }
    }
}
