package cn.com.duibaboot.ext.autoconfigure.data.hbase;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import org.apache.hadoop.hbase.client.HTableInterfaceFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

/**
 * 注入额外的HbaseTemplate
 */
//也可以用BeanDefinitionRegistryPostProcessor来注入BeanDefinition
public class ExtraHbaseImportBeanDefinitionRegistrar implements ImportBeanDefinitionRegistrar, EnvironmentAware {

    private List<String> extraHbaseTemplateIds = new ArrayList<>();

    @Override
    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
        for(String extraHbaseTemplateId : extraHbaseTemplateIds) {
            BeanDefinition extraHbaseTemplateDefinition = BeanDefinitionBuilder
                    .genericBeanDefinition(WrappedHbaseTemplateFactoryBean.class)
                    .addConstructorArgValue(extraHbaseTemplateId)
                    .getBeanDefinition();

            registry.registerBeanDefinition(extraHbaseTemplateId, extraHbaseTemplateDefinition);
        }
    }

    @Override
    public void setEnvironment(Environment environment) {
        String hbaseExtraIdPrefix = "duiba.hbase.extra.";
        String hbaseExtraIdPostfix = ".zk-quorum";
        LinkedHashMap<String, Object> props = SpringEnvironmentUtils.getFlatEnvironments(environment);
        props.entrySet().stream()
                .filter(e -> e.getKey().startsWith(hbaseExtraIdPrefix) && e.getKey().endsWith(hbaseExtraIdPostfix))
                .map(e -> e.getKey().substring(hbaseExtraIdPrefix.length(), e.getKey().length() - hbaseExtraIdPostfix.length()))
                .forEach(s -> extraHbaseTemplateIds.add(s));
    }

    public static class WrappedHbaseTemplateFactoryBean implements FactoryBean<WrappedHbaseTemplate>{

        @Resource
        private HbaseProperties hbaseProperties;
        private String extraKey;

        public WrappedHbaseTemplateFactoryBean(String extraKey){
            this.extraKey = extraKey;
        }

        @Override
        public WrappedHbaseTemplate getObject() throws Exception {
            org.apache.hadoop.conf.Configuration extraHbaseTemplateConfiguration = hbaseProperties.getExtra().get(extraKey).getConfiguration();
            HTableInterfaceFactory extraHTableInterfaceFactory = new HTableConnectionFactory(extraHbaseTemplateConfiguration);

            WrappedHbaseTemplate extraHbaseTemplate = new WrappedHbaseTemplate(extraHbaseTemplateConfiguration);
            //使用自定义的工厂，以维持connection不断开；如果不设置这个，spring官方的HbaseTemplate在每次调用完毕后都会断开长连接.
            extraHbaseTemplate.setTableFactory(extraHTableInterfaceFactory);
            return extraHbaseTemplate;
        }

        @Override
        public Class<WrappedHbaseTemplate> getObjectType() {
            return WrappedHbaseTemplate.class;
        }

        @Override
        public boolean isSingleton() {
            return true;
        }

    }
}
