package cn.com.duibaboot.ext.autoconfigure.datasource;

import cn.com.duibaboot.ext.autoconfigure.perftest.PerfTestRoutingDataSource;
import com.alibaba.fastjson.JSON;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.dbcp2.BasicDataSource;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 用于数据库连接池的实时监控，每秒钟检测一次数据库源的bean，并保存到一个map中，用于endpoint调用
 * Created by guoyanfei .
 * 2018/6/13 .
 */
public class DuibaDataSourceMonitor implements DisposableBean {

    private boolean initted = false;

    private Map<String, BasicDataSource> dataSourceMap;

    private Map<String, PerfTestRoutingDataSource> routingDataSourceMap;

    private Thread duibaDataSourceMonitorThread;

    private static final long RETENTION_TIME = 1000L * 60 * 30;  // 30分钟

    private Map<String, Map<Long, MonitorObj>> secondMonitorMap = new ConcurrentHashMap<>();

    protected synchronized String getSecondMonitorJson() {
        return JSON.toJSONString(secondMonitorMap);
    }

    public synchronized void startMonitorThread(ApplicationContext context) {
        if (!initted) {
            initted = true;
            dataSourceMap = context.getBeansOfType(BasicDataSource.class);
            routingDataSourceMap = context.getBeansOfType(PerfTestRoutingDataSource.class);

            if (dataSourceMap.isEmpty() && routingDataSourceMap.isEmpty()) {
                return;
            }

            duibaDataSourceMonitorThread = new Thread(() -> {
                scanDataSources();
            },"duibaDataSourceMonitorThread");

            duibaDataSourceMonitorThread.start();
        }
    }

    private void scanDataSources(){
        while (true) {
            for (Map.Entry<String, BasicDataSource> entry : dataSourceMap.entrySet()) {
                scanDataSource(entry.getKey(), entry.getValue());
            }
            for (Map.Entry<String, PerfTestRoutingDataSource> entry : routingDataSourceMap.entrySet()) {
                scanDataSource(entry.getKey(), entry.getValue().getOriginalDataSource());
                scanDataSource(entry.getKey() + "_shade", entry.getValue().getShadeDataSource());
            }

            try {
                Thread.sleep(1000);//每隔1S检查一次，记录数据库连接池使用情况
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            if (Thread.currentThread().isInterrupted()) {
                break;
            }
        }
    }

    private synchronized void scanDataSource(String key, BasicDataSource ds) {
        if (ds != null && !ds.isClosed()) {
            int numIdleConnections = ds.getNumIdle(); // 池中有多少个空闲连接，它们可以被checkout
            int numActiveConnections = ds.getNumActive();   // 池中被checkout的连接有多少个
            int maxTotal = ds.getMaxTotal();

            Map<Long, MonitorObj> dbSecondMonitorMap = secondMonitorMap.get(key);
            if (dbSecondMonitorMap == null) {
                dbSecondMonitorMap = new LinkedHashMap<Long, MonitorObj>() {
                    // put的时候，删除30分钟之前put的值
                    @Override
                    protected boolean removeEldestEntry(Map.Entry<Long, MonitorObj> eldest) {
                        return eldest.getKey() < (System.currentTimeMillis() - RETENTION_TIME);
                    }
                };
                secondMonitorMap.putIfAbsent(key, dbSecondMonitorMap);
            }
            dbSecondMonitorMap.put(System.currentTimeMillis(), new MonitorObj(numIdleConnections, numActiveConnections, maxTotal));
        }
    }

    /**
     * 获取数据库连接池配置
     * @return
     */
    public List<DataSourceConfig> getDataSourceConfig() {
        List<DataSourceConfig> dataSourceConfigs = new ArrayList<>();
        for (Map.Entry<String, BasicDataSource> entry : dataSourceMap.entrySet()) {
            BasicDataSource ds = entry.getValue();
            if (ds != null) {
                dataSourceConfigs.add(new DataSourceConfig(entry.getKey(), ds.getUrl(), ds.getUsername(), ds.getMaxTotal(), ds.getInitialSize(), ds.getMaxIdle(), ds.getMinIdle(), ds.getMaxWaitMillis()));
            }
        }
        for (Map.Entry<String, PerfTestRoutingDataSource> entry : routingDataSourceMap.entrySet()) {
            BasicDataSource ds = entry.getValue().getOriginalDataSource();
            if (ds != null) {
                dataSourceConfigs.add(new DataSourceConfig(entry.getKey(), ds.getUrl(), ds.getUsername(), ds.getMaxTotal(), ds.getInitialSize(), ds.getMaxIdle(), ds.getMinIdle(), ds.getMaxWaitMillis()));
            }

            BasicDataSource prds = entry.getValue().getShadeDataSource();
            if (prds != null) {
                dataSourceConfigs.add(new DataSourceConfig(entry.getKey() + "_shade", prds.getUrl(), prds.getUsername(), prds.getMaxTotal(), prds.getInitialSize(), prds.getMaxIdle(), prds.getMinIdle(), prds.getMaxWaitMillis()));
            }
        }
        return dataSourceConfigs;
    }

    @Override
    public synchronized void destroy() throws Exception {
        if (duibaDataSourceMonitorThread != null) {
            duibaDataSourceMonitorThread.interrupt();
        }
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class DataSourceConfig {

        private String dataSourceName;

        private String url;

        private String username;

        private int maxTotal;

        private int initialSize;

        private int maxIdle;

        private int minIdle;

        private long maxWaitMillis;

    }

    public static class MonitorObj {

        private short numActive;
        private short numAllocated;
        private short maxTotal;

        public MonitorObj(int numIdleConnections, int numActiveConnections, int maxTotal) {
            this.numActive = (short) numActiveConnections;
            this.maxTotal = (short) maxTotal;
            this.numAllocated = (short) (numActiveConnections + numIdleConnections);
        }

        public MonitorObj() {
        }

        public short getNumActive() {
            return numActive;
        }

        public short getNumAllocated() {
            return numAllocated;
        }

        public short getMaxTotal() {
            return maxTotal;
        }
    }

}
