package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * 引流回归平台使用，文件上传和下载的组件
 * Created by guoyanfei .
 * 2019-02-22 .
 */
@Slf4j
public class FlowReplayFileComponent {

    private final RestTemplate loadBalancedRestTemplate;

    public FlowReplayFileComponent(RestTemplate loadBalancedRestTemplate) {
        this.loadBalancedRestTemplate = loadBalancedRestTemplate;
    }

    /**
     * 文件上传到oss
     * 1、根据本地文件路径把文件压缩
     * 2、调用athena-web的上传接口上传，返回oss地址
     *
     * @param localPath
     * @return
     */
    public String upload(String localPath) throws IOException {
        String localZipFileName = FlowReplayConstants.LOCAL_FLOWREPLAY_DIR + System.currentTimeMillis() + ".zip";
        File localFile = new File(localPath);
        File localZipFile = new File(localZipFileName);

        // 压缩文件localFile，创建新的文件localZipFile
        byte[] buffer = new byte[1024];
        try (InputStream is = new FileInputStream(localFile); BufferedInputStream bis = new BufferedInputStream(is);
                OutputStream os = new FileOutputStream(localZipFile); ZipOutputStream zipos = new ZipOutputStream(os)) {
            zipos.putNextEntry(new ZipEntry(localFile.getName()));
            int i = bis.read(buffer);
            while (i != -1) {
                zipos.write(buffer, 0, i);
                i = bis.read(buffer);
            }
        }

        // 压缩好的文件localZipFile，通过athena-web上传到oss中
        FileSystemResource resource = new FileSystemResource(localZipFile);
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        multiValueMap.add("file", resource);

        return loadBalancedRestTemplate.postForObject("http://athena-web/athena/file/upload", multiValueMap, String.class);
    }

    /**
     * 把需要回归的文件下载下来
     * 1、根据oss地址，调用athena-web的下载接口下载
     * 2、把下载的文件解压缩
     * @param ossPath
     * @return
     * @throws IOException
     */
    public String download(String ossPath) throws IOException {
        ResponseEntity<Resource> response = loadBalancedRestTemplate.getForEntity("http://athena-web/athena/file/download?objectName=" + ossPath, Resource.class);
        InputStream input = response.getBody().getInputStream();

        // 把下载的zip文件localZipFile解压缩到newFilePath
        String newFilePath = FlowReplayConstants.LOCAL_REPLAY_FILEPATH;
        try (ZipInputStream zipInputStream = new ZipInputStream(input);
                FileOutputStream fileOutputStream = new FileOutputStream(newFilePath)) {
            zipInputStream.getNextEntry();
            byte[] buf = new byte[1024];
            int len;
            while ((len = zipInputStream.read(buf, 0, 1024)) != -1) {
                fileOutputStream.write(buf, 0, len);
            }
        }
        return newFilePath;
    }

    public String uploadAndRemoveDir(String localPath) throws IOException {
        String ossPath = upload(localPath);
        // 上传完成文件删除目录
        removeDir();
        return ossPath;
    }

    public String initDirAndDownload(String ossPath) throws IOException {
        // 删除已有目录，并创建目录
        initDir();
        return download(ossPath);
    }

    /**
     * 初始化引流回归的本地文件目录 /data/flowreplay/
     * 删除，并重新创建目录
     */
    public static void initDir() {
        File dir = new File(FlowReplayConstants.LOCAL_FLOWREPLAY_DIR);
        try {
            FileUtils.deleteDirectory(dir);
            FileUtils.forceMkdir(dir);
        } catch (IOException e) {
            log.error("初始化/data/flowreplay目录异常", e);
        }
    }

    /**
     * 删除引流回归的本地文件目录 /data/flowreplay/
     */
    public static void removeDir() {
        File dir = new File(FlowReplayConstants.LOCAL_FLOWREPLAY_DIR);
        try {
            FileUtils.deleteDirectory(dir);
        } catch (IOException e) {
            log.error("初始化/data/flowreplay目录异常", e);
        }
    }
}
