package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duiba.wolf.entity.Pair;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayEndEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayStartEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer.Hessian2Serializer;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 回归结果写到文件中
 * Created by guoyanfei .
 * 2019-02-26 .
 */
@Slf4j
public class ReplayTraceResultWriter {

    private volatile Thread traceResultWriteThread;

    /**
     * 监听 "回归开始" 事件，把回归结果写到文件中
     * @param event
     */
    @EventListener(ReplayStartEvent.class)
    public void replayStartEventListener(ReplayStartEvent event) {
        traceResultWriteThread = new Thread(new Runnable() {

            @Override
            public void run() {
                ReplayContext context = ReplayContextHolder.getReplayContext();
                if (context == null) {
                    return;
                }
                try {
                    writeTraceResult(context);
                } catch (Exception e) {
                    log.error("引流回归_回归结果写文件异常_reportId_{}", context.getReportId(), e);
                    ReplayContextHolder.forceEnd();
                }
            }
        }, "DBThread-Replay-写回归结果");

        traceResultWriteThread.start();
    }

    @EventListener(ReplayEndEvent.class)
    public void replayEndEventListener(ReplayEndEvent event) {
        if (traceResultWriteThread != null) {
            traceResultWriteThread.interrupt();
        }
    }

    private void writeTraceResult(ReplayContext context) throws IOException {   // NOSONAR
        File localReportFile = new File(FlowReplayConstants.LOCAL_RESULT_FILEPATH);
        File localReplayDetailFile = new File(FlowReplayConstants.LOCAL_REPLAY_DETAIL_FILEPATH);
        try (RandomAccessFile reportFile = new RandomAccessFile(localReportFile, "rw");
                RandomAccessFile replayDetailFile = new RandomAccessFile(localReplayDetailFile, "rw")) {
            reportFile.writeInt(FlowReplayConstants.REPLAY_RESULT_FILE_VERSION);  // 回归结果文件版本
            reportFile.writeInt(0);    // 文件初始化，结果个数为0

            replayDetailFile.writeInt(FlowReplayConstants.RECORD_DETAIL_FILE_VERSION);  // 录制详情文件的版本号
            replayDetailFile.writeInt(0);    // 用例数设置为0
            replayDetailFile.writeLong(0);    // 索引文件的开始位置设置为0

            // 读取整个用例集文件，解析出每个用例
            List<Pair> traceIdIndexList = new ArrayList<>();    // key: traceId, value: index
            int usecaseCount = 0;   // 用例数量
            while (!context.isReplayersDone() || !context.isResultQueueEmpty()) {
                ReplayTraceResult result = null;
                try {
                    result = context.pollResult();
                    if (result == null) {
                        continue;
                    }
                    byte[] resultBytes = Hessian2Serializer.serialize(result);
                    reportFile.writeInt(resultBytes.length);    // 写入单个result的长度，读取的时候先读取长度，然后根据长度，放心读取result
                    reportFile.write(resultBytes);              // 写入result

                    FlowReplayTrace detailTrace = result.getReplayDetailTrace();
                    if (detailTrace != null) {
                        usecaseCount++;

                        Map<String, String> spanStackFramesMap = ReplayContextHolder.getReplayContext().getReplayDetailContext().getAndRemoveTraceSpanStackFrames(detailTrace.getTraceId());
                        detailTrace.convertToDetail(spanStackFramesMap);

                        byte[] traceDetailBytes = Hessian2Serializer.serialize(detailTrace);

                        traceIdIndexList.add(Pair.from(detailTrace.getTraceId(), replayDetailFile.getFilePointer()));
                        replayDetailFile.writeInt(traceDetailBytes.length);
                        replayDetailFile.write(traceDetailBytes);
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } catch (Exception e) {
                    log.error("回归结果操作异常_traceId={}", result != null ? result.getTraceId() : "unknown", e);
                }
                if (Thread.currentThread().isInterrupted()) {
                    break;
                }
            }
            long idxIndex = replayDetailFile.getFilePointer();
            for (Pair p : traceIdIndexList) {
                byte[] traceIdBytes = Hessian2Serializer.serialize(p.getKey());
                replayDetailFile.writeInt(traceIdBytes.length);
                replayDetailFile.write(traceIdBytes);

                replayDetailFile.writeLong((long) p.getValue());
            }
            replayDetailFile.seek(4);
            replayDetailFile.writeInt(usecaseCount); // 重新写入用例数
            replayDetailFile.writeLong(idxIndex);    // 重新写入索引文件的开始位置
        } finally {
            // 文件写好，结束回归
            ReplayContextHolder.end();
        }
    }

}
